/* **********************************************************************
**
**  Copyright (C) 2003-2004	Jesper Hansen <jesperh@telia.com> and 
**				Romuald Bialy (MIS) <romek_b@o2.pl>.
**
*************************************************************************
**
**   This file is part of the yampp system.
**
**  This program is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License
**  as published by the Free Software Foundation; either version 2
**  of the License, or (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software Foundation, 
**  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
*************************************************************************
**
**  yampp-3/USB or yampp-3 with ATmega161/162 CPU. CD-ROM version.
**  Firmware Ver. 1.51 - 2006-05-01
**  For recompliation please use GCC in version 3.4.3 !
**  ( WinAVR-20050214-install.exe  package)
**
*********************************************************************** */

// ------------------------------------------------------------------------
char version_txt[]       __attribute__ ((progmem)) = "1.51";
char date_txt[]          __attribute__ ((progmem)) = __DATE__" / "__TIME__;
char model_txt[]         __attribute__ ((progmem)) = "yampp-3/USB_CD";
char firmware_scan_tag[] __attribute__ ((progmem)) = "yfwTAG";

// ------------------------------------------------------------------------

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <avr/signal.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include <string.h>

#include "Constants.h"
#include "types.h"
#include "mem.h"
#include "delay.h"
#include "atapi.h"
#include "iso9660.h"
#include "vs1001.h"
#include "remote.h"
#include "uart.h"
#include "lcd.h"

// Amount of buffers writed before relinquish control
#define MAX_CONS_WRITES 200

#define	EV_IDLE		0
#define	EV_PLAY		1
#define	EV_STOP		2
#define	EV_NEXT		3
#define	EV_PREV		4
#define	EV_NEXT10	5
#define	EV_PREV10	6
#define	EV_VOLUP	7
#define	EV_VOLDN	8
#define	EV_NEXTD	9
#define	EV_PREVD	10
#define	EV_FFWD		11
#define	EV_FREW		12

#define	EV_LOUDNESS	13
#define	EV_RANDOM	14
#define	EV_REPEAT	15
#define	EV_REMAIN	16
#define	EV_SLEEP	17
#define	EV_EJECT	18
#define	EV_FUNCTION	19
#define	EV_NUMBER	20
#define	EV_PAUSE	21
#define	EV_LASTEVENT	22

// Auxilliary functions declarations
u08* unsigned2str(u16 track, u08 *str);
u08 get_char_event(void);
u08 ir_event(void);
u08 get_key_event(void);
void send_sinewave_beeps(void);
void setvolume(u08 v);
void dispbar(void);
bool RetrieveMP3Data(void);
void BadHeadStrip(void);
u16  Numb2Int(void);
bool LoadCD(void);
void lcd_frame(void);

// Global variables
u08 lcd_steps;
u08 LastEvent	= EV_IDLE;
u08 *scroll_line;
u08 scroll_length;
u08 scroll_pos;
u08 scroll_dir;

#if (LCD_LINES == 4)
 u08 dir_scroll_length;
 u08 dir_scroll_pos;
 u08 *dir_scroll_line;
 u08 dir_scroll_dir;
#endif

bool bError;					// Modificators
extern bool cdda;				// cdda status
bool bPlaying = false;			// Current state - play/stop
bool bTrayOpen;				// Tray closed
u08 *pPlayData;				// Pointer to current position in currently used buffer
u16 wSongNum;				// Current playing song
u16 wMaxSongs;				// Maximum available songs
volatile u08 nDispUpd;			// Counter for display update loop
volatile u08 nKeyTime;			// Counter for keypress time check - Added by MIS
#ifdef ENABLE_SLEEP_TIMER
 volatile u08 nTimer;
 volatile u16 wTimerPresc;
#else
 #define wTimerPresc	0xFFFF
#endif

#ifdef OLD_VS1001
volatile u16 wPlayT;				// Play Time counter for older VS1001 chips
#endif

extern bool cdda;
#if(LCD_LINES == 2)
 u08 *loudstr  = "Loud*";
 u08 *randstr  = "Rand*";
 u08 randmod[] = "-+*";
 u08 tot_sw;
#endif
u08 trackstr[6], *pTrack, timestr[6];
u08 number[5], *pNum = number;

u08 NO_DRIVE[] __attribute__ ((progmem)) = "Drive not found!\r\n";

/************************************************************************
*			TIMER 1 INTERRUPT HANDLER			*
************************************************************************/

#define TI1_VALUE	((u16)-(F_CPU / 2560))

SIGNAL(SIG_OVERFLOW1)							// Timer1 overflow every 100 mS
{
	TCNT1 = TI1_VALUE;							// Reload timer
	nDispUpd++;
	nKeyTime++;
#ifdef OLD_VS1001
	wPlayT++;
#endif
#ifdef ENABLE_SLEEP_TIMER
	if(nTimer)										// if timer ON
	{
		if(wTimerPresc++ == 600)						// 1 min interval
		{
			wTimerPresc = 0;
			if(--nTimer == 0)						// Timer END
			{
				TIMER_PORT &= ~ _BV(TIMER_PIN);		// Power OFF state
				TIMER_DDR |= _BV(TIMER_PIN);		// Short the Timer out PIN to GND
				wTimerPresc = 0xFFFF;			// As timer out pin may be used the IR_IN pin.
			}
		}
	}
#endif
}

/************************************************************************
*			VS1001 EQUALIZER STUFF				*
************************************************************************/

#include "loudness.h"
u16 vs1001_fw = 0;

u16 PRG_RDW(u16 addr)
{
	asm volatile ("lsl  r24");
	asm volatile ("rol  r25");
	asm volatile ("ldi  r30,lo8(dat)");
	asm volatile ("ldi  r31,hi8(dat)");
	asm volatile ("add  r30,r24");
	asm volatile ("adc  r31,r25");
	asm volatile ("LPM  r24,Z+");
	asm volatile ("LPM  r25,Z");
	return addr;
}

void vs1001_loader(void)
{
	// VS1001 User Software Loader
	//

	u16 addr = 2;
	u16 DataLen = PRG_RDW(addr++);				// length of firmware in words (firmware only)
	vs1001_fw = PRG_RDW(addr++);				// address to enter into VS1001 register A1ADDR
											// for application running.
	vs1001_write(7,vs1001_fw + 0x4000);				// write to VS1001 register WRAMADDR
	while(DataLen--)							// data length in words
		vs1001_write(6,PRG_RDW(addr++));			// write firmware word to vs1001
}

void Loud_Set(u08 bLoudness)						// Equalizer mode setting
{
	if(bLoudness)		
	{
		vs1001_write(0x0A,vs1001_fw);			// Enable user code
		vs1001_write(0x0D,0);				// Reset EQ
		vs1001_write(0x0D,bLoudness);			// Set EQ characteristic
	}
	else
	{
		vs1001_write(0x0D,0);				// Reset EQ
		vs1001_write(0x0A,0);				// Disable user code
	}
}


u08 ldchar(u08 mode)
{
	return (mode>9) ? mode+('A'-10) : mode+'0';
}

/************************************************************************
*				RANDOM STUFF				*
************************************************************************/

// Random stuff, maximum 4096 songs on the disk!

u32 seed = 1;

// Init randomize table
void InitRnd(void)
{
	memset(RANDOM_TAB,0,512);
}

u16 randcalc(u16 max)					// assembler optimized randomizer function
{
//  return ((seed = seed * 1103515245L + 12345) % max);	// get new song number

	asm volatile ("push r24");
	asm volatile ("push r25");
	asm volatile ("lds r22,seed");
	asm volatile ("lds r23,(seed)+1");
	asm volatile ("lds r24,(seed)+2");
	asm volatile ("lds r25,(seed)+3");
	asm volatile ("ldi r18,lo8(1103515245)");
	asm volatile ("ldi r19,hi8(1103515245)");
	asm volatile ("ldi r20,hlo8(1103515245)");
	asm volatile ("ldi r21,hhi8(1103515245)");
	asm volatile ("call __mulsi3");
	asm volatile ("subi r22,lo8(-(12345))");
	asm volatile ("sbci r23,hi8(-(12345))");
	asm volatile ("sbci r24,hlo8(-(12345))");
	asm volatile ("sbci r25,hhi8(-(12345))");
	asm volatile ("sts seed,r22");
	asm volatile ("sts (seed)+1,r23");
	asm volatile ("sts (seed)+2,r24");
	asm volatile ("sts (seed)+3,r25");
	asm volatile ("pop r19");
	asm volatile ("pop r18");
	asm volatile ("clr r20");
	asm volatile ("clr r21");
	asm volatile ("call __udivmodsi4");
	asm volatile ("mov r25,r23");
	asm volatile ("mov r24,r22");
	return max;
}

u16 wFirstInDir = 0;

u08 tabval(u16 num, u08 fn)					// check random tabele for song status
{								// and mark as played if necessary
	u16 index = (num+wFirstInDir) >> 3;
	u08 offset = 1 << ((num+wFirstInDir) % 8);
	if(fn)
		RANDOM_TAB[index] |= offset;			// if fn=1 mark as played
	return (RANDOM_TAB[index] & offset);			// return status
}

// new random procedure
u16 do_rand(u16 max)						// returned random value from 0 to max - 1
{
	u16 num;
	max &= 0x0fff;						// limit to 4096 songs for lower RAM usage
	for(num = 0; num < max; num++)
	{
		if(tabval(num,0) == 0)
			break;
	}
	if (num == max)
	{
		InitRnd();
		num = -1;
	}
	else
	{
		do
		{
			wdt_reset();
			num = randcalc(max);
		} while (tabval(num,0) != 0);			// check this song not be played piervously
		tabval(num,1);					// mark as just played
	}
	return num;
}

//----------------------------------------------------------------------------
// Helper functions
//----------------------------------------------------------------------------

// This function return number of songs in current direcrory and sets first song number variable
u16 checkdir(void)
{
	u08 Temp;
	for (Temp = 0; Temp < nLastPosD; Temp++)			// search dir tabele
	{
		if (DirsTable[Temp].song_num <= wSongNum)
			wFirstInDir = DirsTable[Temp].song_num;		// update first song number in current dir
		else
			break;						// found end of current dir
	}
	if (Temp == nLastPosD)						// if this is last dir on CD
		return (wMaxSongs - wFirstInDir);
	return (DirsTable[Temp].song_num - wFirstInDir);
}


//----------------------------------------------------------------------------

void printsp(void)
{
	register u08 i= LCD_LINE_LENGTH;
	while(i--) lcd_putchar(' ');
}

void print_OFF(void)
{
	lcd_puts_p(PSTR("OFF "));
}

//----------------------------------------------------------------------------

void LCD_Info(void)
{
	lcd_clrscr();
#if (LCD_LINE_LENGTH == 16)
	lcd_puts_p(PSTR("YAMPP MP3 Player\n"));
#else
	lcd_puts_p(PSTR(" YAMPP-3 MP3 Player\n"));
#endif
}

//----------------------------------------------------------------------------

void LCD_DiscTime(void)
{
	udiv_t divt;
	u08 *src;
	u16 tmp = ATAPI_DA_GetDiscTime();
	src = trackstr;
	divt = udiv(tmp, 600);
	*src++ = '0' + (u08)divt.quot;
	divt = udiv(divt.rem, 60);
	*src++ = '0' + (u08)divt.quot;
	*src++ = ':';
	divt = udiv(divt.rem, 10);
	*src++ = '0' + (u08)divt.quot;
	*src++ = '0' + (u08)divt.rem;			
	*src = 0;
	lcd_puts(trackstr);			// display disc time
}

//----------------------------------------------------------------------------

#ifdef ENABLE_SLEEP_TIMER
void TimerDisplay(void)
{
	u08 *src;
	udiv_t divt;
	nDispUpd = 230;
	lcd_gotoxy(0,LCD_LINES - 1);
	lcd_puts_p(PSTR("Sleep: "));
	if (nTimer)
	{
		src = timestr;
		divt = udiv(nTimer, 60);
		*src++ = '0' + (u08)divt.quot;
		*src++ = ':';
		divt = udiv(divt.rem, 10);
		*src++ = '0' + (u08)divt.quot;
		*src++ = '0' + (u08)divt.rem;			
		*src = 0;
		lcd_puts(timestr);			// display timer value
	}
	else
		print_OFF();
	printsp();
}
#endif

//----------------------------------------------------------------------------

void Do_Eject(void)
{
	vs1001_reset();
	bPlaying = false;
	wMaxSongs = 0;
	LCD_Info();
	if (!bTrayOpen)
	{
		lcd_puts_p(PSTR("Open"));
		CD_Cmd(EJECT_CD);
		bTrayOpen = true;
	}
	else
		LoadCD();
}

//----------------------------------------------------------------------------

u16 GetSongTime(u08 songnum)
{
	u16 tmp1 = (60 * ATAPI_DA_GetM(songnum)) + ATAPI_DA_GetS(songnum);
	u16 tmp2 = (60 * ATAPI_DA_GetM(songnum+1)) + ATAPI_DA_GetS(songnum+1);
	return(tmp2-tmp1);
}

//----------------------------------------------------------------------------
// Main part
//----------------------------------------------------------------------------
int main(void)
{
	u08 i;
#ifndef SETUP_REMOTE_CODES
	u08 bRandom    = 0;				// Modificators
	u08 bRepeat    = 0;				// Modificators
	u08 bLoudness  = 0;				// Modificators
	bool bFfwdMute = false;			// Search mute flag
	bool bTimeMode = false;			// Remain time modificator
	u08 nVolume = 0;				// Current volume level
	u16 wPlayTime = 0;				// Current play time
	u16 wSongTime = 0;				// Song time for remain calculation
	u16 wBitrate = 0;				// Song Bitrate
	u16 wFilteredBitrate = 0;			// Filtered bitrate for time calculations
	u16 wAddTime = 0;				// Fix play time for search
	u08 nWrites, nBursts;			// Buffering variables
	u08 event = EV_IDLE;			// Current event
	u08 *src, *dst;					// Temporary variables
	u16 wTemp;
	udiv_t divt;
	u08 function = 0;				// For Local Keyboard menu
#endif
//----------------------------------------------------------------------------
// B - Port
//----------------------------------------------------------------------------
	PORTB = 0x57;
	DDRB = 0xBB;
	// PB0 is ATA DIOW,		OUT-HIGH
	// PB1 is ATA DIOR,		OUT-HIGH
	// PB2 is VS1001 DREQ,		IN-PULLUP
	// PB3 is VS1001 BSYNC,		OUT-LOW
	// PB4 is VS1001 MP3,		OUT-HIGH
	// PB5 (MOSI) is used for SPI,	OUT-LOW
	// PB6 (MISO) is used for SPI,	IN-PULLUP
	// PB7 (SCK) is used for SPI,	OUT-LOW

#ifdef YAMPP3USB
//----------------------------------------------------------------------------
// D - Port on yampp-3/USB
//----------------------------------------------------------------------------
	PORTD = 0xFB;
	DDRD = 0xC4;
	// PD0 is RXD
	// PD1 is TXD
	// PD2 is MP3/CD-AUDIO selector,	OUT-LOW
	// PD3 is keyboard input,		IN-PULLUP
	// PD4 is TXF for FTDI,			IN-PULLUP
	// PD5 is TXF for FTDI,			IN-PULLUP
	// PD6 and PD7 are RD and WR		OUT-HIGH
	// signals but we need to set their default states used when the
	// external memory interface is disabled they should then both be
	// output and high to avoid disturbing RAM contents
#else
//----------------------------------------------------------------------------
// D - Port on old yampp-3 board
//----------------------------------------------------------------------------
	PORTD = 0xCF;
	DDRD = 0xF0;
	// PD0 is RXD
	// PD1 is TXD
	// PD2 is IR input,			IN-PULLUP
	// PD3 is keyboard input,		IN-PULLUP
	// PD4 is MP3/CD-AUDIO selector,	OUT-LOW
	// PD5 is LCD_ENABLE,			OUT-LOW
	// PD6 and PD7 are RD and WR		OUT-HIGH
	// signals but we need to set their default states used when the
	// external memory interface is disabled they should then both be
	// output and high to avoid disturbing RAM contents
#endif

//----------------------------------------------------------------------------
// E - Port
//----------------------------------------------------------------------------
	PORTE = 0x01;
	DDRE   = 0x06;
	// PE0 is IR input in y3/u, activate pullup
	// PE1 is ALE pin as output, and set it low (inactive)
	// PE2 is LCD_ENABLE in y3/u, and set it low (inactive)

//----------------------------------------------------------------------------
// Timer 1
//----------------------------------------------------------------------------
	// setup timer1 for a 100mS periodic interrupt
	TCCR1A = 0;
	TCCR1B = 4;				// Prescaler /256  tPeriod = 32 uS
	TCNT1 = TI1_VALUE;		// Load counter value
	
//----------------------------------------------------------------------------
// Initialization
//----------------------------------------------------------------------------
	MCUCR = (1<<SRE) | (1<<SRW10);		// enable external SRAM and wait states 
	SPCR = (1<<MSTR) | (1<<SPE);		// SPI 2 MHz clock
	i = SPSR;							// clear SPI status
	delayms(200);
#ifdef ENABLE_WATCHDOG				// Enable watchdog (enabled by linker option)
	wdt_enable(0x07);
#endif
	UART_Init();						// Init RS-232 link
	PRINT_p(PSTR("\nHi! yampp-3/u player CD version\n"));
   	lcd_init(0, LCD_FUNCTION_DEFAULT);
	LCD_Info();
	vs1001_setcomp(F_VS1001);			// set frequency of vs1001
	vs1001_reset();						// Init VS1001

#ifdef SETUP_REMOTE_CODES
	while(1)
		remote_setup();					// go to remote setup procedure
#else

	TIMSK |= _BV(TOIE1);	 			// Enable timer1 interrupt
	if(ATA_Init())						// Init and wait for drive
	{
		PRINT_p((u08*)&NO_DRIVE[0]);
		lcd_puts_p((u08*)&NO_DRIVE[0]);
		while(1);
	}

	nVolume = eeprom_read_byte((u08*)EEPROM_VOLUME);	// Load and set initial volume
	if (nVolume > MIN_VOLUME)
		nVolume = MIN_VOLUME / 2;	// set volume to 1/2 scale after eeprom erase (reprograming chip) :-)

	vs1001_loader();
	setvolume(nVolume);
	send_sinewave_beeps();			// Send three sinewave beeps to indicate
	bRandom   = eeprom_read_byte((u08*)EEPROM_RANDOM);	// options settings
	bLoudness = eeprom_read_byte((u08*)EEPROM_LOUDNESS)%13;
	bTimeMode = eeprom_read_byte((u08*)EEPROM_TIMEMODE);
	bRepeat   = eeprom_read_byte((u08*)EEPROM_REPEAT);
#if (LCD_LINES == 2)
	*(randstr+4) = randmod[bRandom];	// set initial status in the strings
	*(loudstr+4) = (bLoudness) ? bLoudness+'0' : '-';
#endif
	*pNum = 0;
	delayms(1000);
	LoadCD();
	seed = 1 + TCNT1;				// Init randomizer from timer

//----------------------------------------------------------------------------
// start things rolling
//----------------------------------------------------------------------------
	while (true)					// Main loop
	{
		wdt_reset();					// Give watchdog full time
		if (UART_HasChar())				// Check all input controls
			event = get_char_event();

		if (rec80_active() && wTimerPresc != 0xFFFF)
			event = ir_event();
		else
			if (nKeyTime > 16)
				LastEvent = EV_IDLE;

		if (event == EV_IDLE)
			event = get_key_event();

		if (bPlaying && !function && event == EV_PLAY)	// second play press does pause
			event = EV_PAUSE;

#ifdef ENABLE_SLEEP_TIMER
		if(wTimerPresc == 0xFFFF)
		{
			lcd_clrscr();
			lcd_puts_p(PSTR("Good Bye !!!"));
			delayms(5000);
			event = EV_STOP;
		}
#endif

		switch (event)				// According input control
		{
			case EV_IDLE:
				break;

			case EV_PREVD:				// Switch to previous start dir
				if (!cdda)
				{
					checkdir();
					if(wFirstInDir == 0)
						wSongNum = wMaxSongs - 1;
					else
						wSongNum = wFirstInDir - 1;

					checkdir();
					wSongNum = wFirstInDir;
					goto PLAY;
				}
				break;

			case EV_NEXTD:				// Switch to next dir
				if (!cdda)
				{
					wSongNum = checkdir() + wFirstInDir; 
					goto PLAY;
				}
				break;

			case EV_NUMBER:
				pTrack = number;
#if (LCD_LINES == 2)
				tot_sw = 10;
#endif
				goto DISPUPD;

			case EV_PREV10: 
				wSongNum -= 10;	
				goto PLAY;

			case EV_NEXT10:	
				wSongNum += 10;	
				goto PLAY;

			case EV_PREV:
#ifdef NEW_PREV
 				if(dwBytesLoaded > ((cdda) ? 0x0200 : 0xE000))
					goto PLAY;
#endif
				if (bRandom == 0)			// in normal mode
				{
					if (wSongNum == 0)
						wSongNum = wMaxSongs - 1;
					else
						wSongNum--;
					goto PLAY;
				}
				else 
					goto RAND;			// in any random mode

			case EV_NEXT:
NEXT_SONG:		if (bRandom == 0)
				{
					wSongNum++;			// in normal mode
					goto PLAY;
				}
RAND:
				if(cdda && (bRandom > 1))
					bRandom = 1;

				// reinit randomizer from timer - random function work better.
				seed += TCNT1 + 1;

				switch (bRandom)
				{
					case 1:
						wFirstInDir = 0;
						wSongNum = do_rand(wMaxSongs);		// random from all songs on CD
						break;

					case 2:	wTemp = do_rand(checkdir());		// random from current directory
						if (wTemp == 0xFFFF)				// no more songs in current dir.
						{
							if(bRepeat == 2)
								InitRnd();
							else
							{
								wTemp = checkdir();		// calculate number of first song
								wSongNum = wTemp + wFirstInDir;	// in next directory
								if (wSongNum >= wMaxSongs)
									wFirstInDir = wSongNum = 0;
							}
							wTemp = do_rand(checkdir());	// random from this directory
						}
						wSongNum = wTemp + wFirstInDir;		// set new song number
						break;
				}
				goto PLAY;

			case EV_PLAY:
				if (function)
				{
					event = function+12;
					nDispUpd = 240;
					continue;
				}

PLAY:			function = 0;
				if (wMaxSongs == 0)
				{
					if (LoadCD())
						break;
				}

				LCD_Info();
				lcd_puts_p(PSTR("Loading..."));
				if (!bPlaying)
				{
					lcd_gotoxy(LCD_LINE_LENGTH-4,LCD_LINES-1);
					lcd_puts_p(PSTR("PLAY"));
				}

				if (wSongNum >= wMaxSongs)		// Protect for song number error
						wSongNum = 0;

				if (cdda)
				{
					ATAPI_DA_Play(wSongNum);
					delayms(100);

					if (!bPlaying)
						delayms(600*DELAY_TIME);

					*LONG_NAME = 0;
					*DIR_NAME = 0;
					strcpy(LONG_NAME,"Playing ");
					scroll_line = (strcat(LONG_NAME,CD_NAME));
					wSongTime = GetSongTime(wSongNum);
					dwFileSize = (u32)wSongTime << 8;
				}
				else
				{
					if (!bPlaying)
					{
						ATAPI_ReadLBA(ISO_BOOT_BLOCK,BUFFER1P);
						delayms(500);			// CD spinup
					}
					bPlaying = false;
					vs1001_reset();				// Mandatory!
					Loud_Set(bLoudness);
					if (!get_dir_entry(wSongNum))		// Setup buffer for play
					{
						wSongNum = 0;			// Something wrong
						continue;
					}
				}
				PRINT_p(PSTR("\nPlaying #"));
				pTrack = unsigned2str(wSongNum + 1, trackstr);
				PRINT(pTrack);
				UART_SendByte(' ');
#if (LCD_LINES == 2)
				scroll_line = strcat(DIR_NAME, LONG_NAME);
				if(cdda)
					lcd_steps = LCD_LINE_LENGTH - 6;
				else
					lcd_steps = LCD_LINE_LENGTH - 7 - 2*(LCD_LINE_LENGTH != 16);
#else
				if (*DIR_NAME == 0 && !cdda)
					strcpy(DIR_NAME,"ROOT");

				scroll_line = LONG_NAME;
				dir_scroll_line = DIR_NAME;
				dir_scroll_length = strlen(DIR_NAME);
				dir_scroll_pos = dir_scroll_dir = 0;
				lcd_steps = LCD_LINE_LENGTH - 6;
#endif
				scroll_pos = scroll_dir = 0;
				scroll_length = strlen(scroll_line);
				lcd_clrscr();
#if (LCD_LINES == 2)
				PRINT(scroll_line);				// directory and file name
				lcd_puts(scroll_line);
#endif
#ifndef ALTERNATE_SCROLL
				if (scroll_length > LCD_LINE_LENGTH)
				{
					strcat(scroll_line, " >> ");		// add ' >> ' sign
					scroll_length += 4;		
				}
#endif
#if (LCD_LINES == 4)
				// directory name centering

				lcd_gotoxy(0, 0);

				if(cdda)
					LCD_Info();
				else
				{
					if(dir_scroll_length < LCD_LINE_LENGTH)	// only if no dir name scrolling
					{
						for(i=0; i<LCD_LINE_LENGTH/2 - (dir_scroll_length+1)/2; i++)
							lcd_putchar('-');
					}

					lcd_puts(DIR_NAME);			// display directory name

					if(dir_scroll_length < LCD_LINE_LENGTH)	// only if no dir name scrolling
					{
						for(i=0; i<LCD_LINE_LENGTH/2 + 1; i++)
							lcd_putchar('-');
					}
				}
				PRINT(DIR_NAME);
				PRINT_p(PSTR(" / "));
				lcd_gotoxy(0,1);			// Second line = filename
				lcd_puts(LONG_NAME);			// Just display it 
				PRINT(LONG_NAME);
#endif

PLAY_END:
				dwBytesLoaded=0;
				if(!cdda)
				{
					wSongTime = 0;
					wBitrate = wFilteredBitrate = 0;
					bBuf1Empty = bBuf2Empty = true;
					RetrieveMP3Data();
					pPlayData = BUFFER1P;
#ifdef OLD_VS1001
					wPlayT = 0;
					BadHeadStrip();
#endif
				}
				wAddTime = 0;
				eeprom_write_word((u16*)EEPROM_SONG_NUMBER , wSongNum);
				if(bRandom)
					tabval(wSongNum,1);			// mark as played in random mode
				nDispUpd = (cdda) ? 0 : 10;
#if (LCD_LINES == 2)
				tot_sw = 10;
#endif
				bPlaying = true;
				lcd_frame();
				break;

			case EV_STOP:
				bPlaying = false;
				function = 0;
				LCD_Info();
				lcd_puts(CD_NAME);
				pTrack = unsigned2str(wMaxSongs,trackstr);
				strcpy(LONG_NAME,pTrack);
				if(cdda)
				{
					ATAPI_DA_Stop();
#if (LCD_LINES == 2)
					lcd_puts_p(PSTR(", "));
#else
					lcd_putchar('\n');
					strcat(LONG_NAME," tracks, ");
					lcd_puts(LONG_NAME);
#endif
					LCD_DiscTime();
				}
				else
				{
					vs1001_reset();
#if (LCD_LINES == 4)
					lcd_puts_p(PSTR("\nTotal: "));
					strcat(LONG_NAME," songs.");
					lcd_puts(LONG_NAME);
#endif
				}
				CD_Cmd(STOP_CD);
#if (LCD_LINES == 4)
				lcd_gotoxy(LCD_LINE_LENGTH-4,3);
				lcd_puts_p(PSTR("STOP"));
#endif
				break;

			case EV_EJECT:
				Do_Eject();
				break;

			case EV_VOLDN:
				if (!cdda)
				{
					if (nVolume <= MIN_VOLUME-VOL_STEP)
						nVolume += VOL_STEP;
					else
						nVolume = MIN_VOLUME;
					goto SETVOL;
				}
				break;

			case EV_VOLUP:
				if (!cdda)
				{
					if (nVolume >= VOL_STEP)
						nVolume -= VOL_STEP;
					else
						nVolume = 0;
SETVOL:					setvolume(nVolume);
				}
				break;

			case EV_LOUDNESS:
				if(!cdda)
				{
					bLoudness = (bLoudness+1) % 13;
#if (LCD_LINES == 2)
					*(loudstr+4) = (bLoudness) ? ldchar(bLoudness) : '-';
#endif
					Loud_Set(bLoudness);
					eeprom_write_byte((u08*)EEPROM_LOUDNESS, bLoudness);
					if(function)
					{
//						nDispUpd = 240;
						lcd_gotoxy(9,LCD_LINES-1);
						if(bLoudness)
						{
							lcd_putchar('L');
							lcd_putchar(ldchar(bLoudness));
							printsp();
						}
						else
							print_OFF();
					}
					else
					{
#if (LCD_LINES == 2)
						tot_sw = 14;
#endif
						goto DISPUPD;
					}
				}
				break;

			case EV_REMAIN:
				bTimeMode = !bTimeMode;
				eeprom_write_byte((u08*)EEPROM_TIMEMODE, bTimeMode);
				if(function)
				{
					lcd_gotoxy(5,LCD_LINES-1);
					lcd_puts_p((bTimeMode) ? PSTR("Remain") : PSTR("Normal"));
				}
				else
				{
#if (LCD_LINES == 2)
					tot_sw = 0;
#endif
					goto DISPUPD;
				}
				break;
				
			case EV_REPEAT:
				if(nDispUpd > 127)
				{
					bRepeat = (bRepeat + 1) % 3;				// set repeat mode from 0 to 2
					eeprom_write_byte((u08*)EEPROM_REPEAT, bRepeat);		// 0 = normal
				}
				nDispUpd = 240;
				lcd_gotoxy(0,LCD_LINES-1);
				lcd_puts_p(PSTR("Repeat: "));
				switch(bRepeat)
				{
					case 0: print_OFF();
						break;
					case 1: lcd_puts_p(PSTR("SONG"));
						break;
					case 2:
						if(cdda)
							lcd_puts_p(PSTR("ALL "));
						else
						{
							lcd_puts_p(PSTR("DIR "));
							if(bRandom)
							{
								bRandom = 2;
#if (LCD_LINES == 2)
								*(randstr+4) = '*';
#endif
							}
						}
						break;
				}
				printsp();
				break;

			case EV_RANDOM:
				bRandom = (bRandom + 1) % ((cdda)? 2 : 3);		// set random mode from 0 to 2
				if(bRepeat == 2 && bRandom)
					bRandom = 2;
#if (LCD_LINES == 2)

				*(randstr+4) = randmod[bRandom];
#endif
				eeprom_write_byte((u08*)EEPROM_RANDOM, bRandom);
				if(function)
				{
					lcd_gotoxy(7,LCD_LINES-1);
					switch(bRandom)
					{
						case 0: print_OFF();
							break;
						case 1: lcd_puts_p(PSTR("ALL "));
							break;
						case 2: lcd_puts_p(PSTR("DIR "));
							break;
					}
				}
				else
				{
#if (LCD_LINES == 2)
					tot_sw = 12;
#endif
DISPUPD:				nDispUpd = 10;
				}
				break;
				
			case EV_PAUSE:
				event = EV_IDLE;
				if(cdda)
					ATAPI_DA_PauseResume(0);
#if (LCD_LINES == 2)
				lcd_gotoxy(LCD_LINE_LENGTH - 6 - 2*(LCD_LINE_LENGTH != 16),1);	// 2 line
				lcd_puts_p(PSTR("Paused"));
#else
				lcd_gotoxy(0,2);			// 3 line
				lcd_puts_p(PSTR("Paused"));
				printsp();
#endif
				while (rec80_active()) ;	// wait for key release
				do
				{
					wdt_reset();				// Give watchdog full time
					if (UART_HasChar())			// Check all input controls
						event = get_char_event();
					if (rec80_active())
						event = ir_event();
					if (event == EV_IDLE)
						event = get_key_event();
				} while (event == EV_IDLE);
#if (LCD_LINES == 4)
//				lcd_frame();
#endif // LCD_LINES == 4

				if(cdda)
					ATAPI_DA_PauseResume(1);

				break;

			case EV_FFWD:
				if(cdda)
				{
					ATAPI_DA_Scan(0);
					wPlayTime = ATAPI_DA_GetCurTime(); 
					dwBytesLoaded = (u32)wPlayTime << 8;
				}
				else
				{
					for (i = 0; i < 48; i++)			// Number of sectors to skip 6 second
					{
						dwCurrentBlock++;
						dwBytesLoaded += BPS;
						if (!CheckAndAdvanceBlock()) 		// Skip block
							goto NEXT_SONG;			// if end of file reached
					}
SEARCHGO:				vs1001_setvolume(nVolume+40, nVolume+40);	// Decrease volume by 20dB for
											// eliminate sound scratches
					wAddTime = (u16)(dwBytesLoaded / (wFilteredBitrate * 125))
						   - vs1001_read(VS1001_PLAYTIME);
											// Actual SongTime calculation
											// on VBR files its imposible
					bFfwdMute = true;
					bBuf1Empty = true;
					RetrieveMP3Data();				// Load buffer
					pPlayData = BUFFER1P;
 #ifdef OLD_VS1001
					BadHeadStrip();
 #endif
				}
#if (LCD_LINES == 2)
				tot_sw = 0;
#endif
				goto DISPUPD;

			case EV_FREW:
				if(cdda)
				{
					ATAPI_DA_Scan(1);
#if (LCD_LINES == 2)
					tot_sw = 0;
#endif
					wPlayTime = ATAPI_DA_GetCurTime(); 
					dwBytesLoaded = (u32)wPlayTime << 8;
					goto DISPUPD;
				}
				else
				{
					if (dwBytesLoaded >= 0x18000)
					{
						dwCurrentBlock -= 48;
						dwBytesLoaded -= 0x18000;
						goto SEARCHGO;
					}
				}
				break;

#ifdef ENABLE_SLEEP_TIMER
			case EV_SLEEP:
				if(nDispUpd > 127)
				{
					i = (nTimer > 80) ? 30:15;
					nTimer = (nTimer/i)*i + i;
					wTimerPresc = 0;
				}
				if(nTimer > 220)
					nTimer = 0;

				TimerDisplay();
				break;
#endif

			case EV_FUNCTION:
				function = (function+1) % 6;
				if(!function)
				{
					if(bPlaying)
						function = 1;
					else
					{
						event = EV_STOP;
						continue;
					}
				}
				nDispUpd = 230;
				lcd_gotoxy(0,LCD_LINES-2);
				lcd_puts_p(PSTR("Select Function:"));
				printsp();
				lcd_putchar('\n');
				switch(function)
				{
					case 1: lcd_puts_p(PSTR("Loudness mode"));
						break;
					case 2: lcd_puts_p(PSTR("Random mode"));
						break;
					case 3: lcd_puts_p(PSTR("Repeat mode"));
						break;
					case 4: lcd_puts_p(PSTR("Time mode"));
						break;
					case 5: lcd_puts_p(PSTR("Sleep timer"));
						break;
				}
				printsp();
				break;

			default:
				break;
		} // switch (event)

		event = EV_IDLE;			// We always go back to IDLE when
							//  exiting switch


		if (bPlaying)
		{
			if (LastEvent == EV_NUMBER && (nKeyTime > 14 || pNum == number+4))
			{
				LastEvent = EV_IDLE;
				wTemp = Numb2Int();
				pNum = number;
				*pNum = 0;
				if (wTemp <= wMaxSongs)
				{
					wSongNum = wTemp-1;
					goto PLAY;
				}
				pTrack = unsigned2str(wSongNum + 1, trackstr);
			}

//------------------------- SENDING DATA TO VS1001 ------------------------------------------------			

			wdt_reset();						// Enable time to push data

			if(!cdda)
			{
				nWrites = 0;					// This count prevents indefinite
				do							//  feed of data to VS1001 with no
				{							//  way to get control
	
					nBursts =  0;					// Amount of rest of data bursts
	
					// While we have not a to fast start and  DREQ is high and a correct END
					while (bit_is_set(PINB, PB2) && (nBursts < BUFFER_SIZE/32) &&
						(!(bBuf1Empty && bBuf2Empty)))		// Change by MIS 25.06
					{
						vs1001_send_32(pPlayData);		// Feed data to VS1001
						pPlayData += 32;			// Advance pointer
						nBursts++;
						if (pPlayData==BUFFER2P+BUFFER_SIZE) 	// End of Buffer 2 TH
						{
							pPlayData = BUFFER1P;		// Pointer to Start TH
							bBuf2Empty = true;		// Buffer2 is free TH
						}
						if (pPlayData==BUFFER1P+BUFFER_SIZE)	// End of Buffer 1 TH
							bBuf1Empty = true;		// Buffer1 is free TH
					}
				}while( ++nWrites < MAX_CONS_WRITES  &&			// We write not too many times
					RetrieveMP3Data());
	
				if (nWrites >= MAX_CONS_WRITES)				// if vs1001 needs to more data
				{
					vs1001_reset();					// try resetting it
					Loud_Set(bLoudness);
				}
	
				if (bError)
				{
					UART_Puts_p(PSTR("\nRead ERROR !"));
					bError = 0;					// Check for error on CD or manual tray open
					if (ATAPI_CheckStatus() == CD_STAT_OPENED)
					{
						Do_Eject();
						event = EV_IDLE;			// Media removed
					}
					continue;
				}
				if (bFfwdMute && (nDispUpd == 4))			// If fast forward volume downed
				{
					vs1001_setvolume(nVolume,nVolume);		// Restore volume
					bFfwdMute = false;
				}

				if (bBuf1Empty && bBuf2Empty)				// No more data to play
				{							// Try to autoplay next song
					if (bRepeat == 1)
						goto PLAY;

					else if(bRepeat == 2 && !bRandom && (wSongNum == checkdir() + wFirstInDir - 1))
					{
						wSongNum = wFirstInDir; 
						goto PLAY;
					}

					else
						event = EV_NEXT;

					continue;
				}
			}	

// ------------------------ DISPLAY NAMES SCROLLING ---------------------------------------------------------


			if (nDispUpd >= 5 && nDispUpd < 200)			// Update display info every 500ms
			{
				if(bLoudness)
					vs1001_write(0x0D,bLoudness);		// Set EQ characteristic if VS1001 selfreset

				function = 0;

				if (scroll_length > LCD_LINE_LENGTH)
				{
					lcd_gotoxy(0, LCD_LINES/2 - 1);
					for (wTemp = 0; wTemp < LCD_LINE_LENGTH; wTemp++)
#ifdef ALTERNATE_SCROLL
						lcd_putchar(scroll_line[scroll_pos + wTemp]);
	
					if (scroll_pos + LCD_LINE_LENGTH == scroll_length)
						scroll_dir -= 2;

					if (scroll_pos == 0)
						scroll_dir++;

					scroll_pos += (signed char)scroll_dir / 4;

#else
						lcd_putchar(scroll_line[(scroll_pos + wTemp) % scroll_length]);
					scroll_pos++;
#endif
				}

// ------------------------ 4 LINES LCD , Additional DIR Name Scrolling ----------------------------------------

#if (LCD_LINES == 4)

				if (dir_scroll_length > LCD_LINE_LENGTH)
				{
					lcd_gotoxy(0, 0);
					for (wTemp = 0; wTemp < LCD_LINE_LENGTH; wTemp++)
#ifdef ALTERNATE_SCROLL
						lcd_putchar(dir_scroll_line[dir_scroll_pos + wTemp]);

					if (dir_scroll_pos + LCD_LINE_LENGTH == dir_scroll_length)
						dir_scroll_dir -= 2;

					if (dir_scroll_pos == 0)
						dir_scroll_dir++;

					dir_scroll_pos += (signed char)dir_scroll_dir / 4;

#else //ALTERNATE_SCROLL
						lcd_putchar(dir_scroll_line[(dir_scroll_pos + wTemp) % dir_scroll_length]);

					dir_scroll_pos++;
#endif //ALTERNATE_SCROLL
				}
#endif // LCD_LINES == 4

				if (! cdda)
				{
					if((wTemp = vs1001_read(VS1001_AUDATA) & 0x01ff) != 0)	// read AUDATA register
					{
						wBitrate = wTemp;
						if(dwBytesLoaded < 0x30000 || wFilteredBitrate == 0)
							wFilteredBitrate = wTemp;
						else
							wFilteredBitrate = (wFilteredBitrate*199 + wBitrate) / 200;
											// Filtered bitrate on VBR songs

						wSongTime = (u16)(dwFileSize / (wFilteredBitrate * 125)); 
											// Song time calculation
					}
				}

//------------------------ TIME, TRACK NUMBER AND OTHER DISPLAYING   ( 2 LINES LCD ) -------------------------

				dispbar();
				dst = timestr;
#if (LCD_LINES == 2)
				lcd_gotoxy(lcd_steps, 1);		// Line 2

				if (tot_sw < 10)	// 0-9
				{
					src = timestr;
					if (!cdda)
					{
#ifdef OLD_VS1001
						wPlayTime = wPlayT / 10 + wAddTime;	// seconds
						if (wPlayTime > 32768) wAddTime = -(wPlayT / 10);
#else  //OLD_VS1001
						wPlayTime = vs1001_read(VS1001_PLAYTIME) + wAddTime;
#endif  //OLD_VS1001
					}
					else if (ATAPI_DA_GetSubC() == 0)
					{
						wTemp = ATAPI_DA_GetCurTrack();
						if(wTemp != wSongNum)		// if playing next song
						{
							if(bRandom || bRepeat == 1)
								goto RAND;

							wSongNum = wTemp;
							pTrack = unsigned2str(wSongNum + 1, trackstr);
							wSongTime = GetSongTime(wSongNum);
							dwFileSize = (u32)wSongTime << 8;
							goto PLAY_END;
						}
						else
						{
							wPlayTime = ATAPI_DA_GetCurTime(); 
							dwBytesLoaded = (u32)wPlayTime << 8;
						}
					}
					else if (ATAPI_CheckStatus() == CD_STAT_OPENED)
					{
						Do_Eject();
						event = EV_IDLE;					// Media removed
						continue;
					}
					else if (ATAPI_DA_GetSubC() >= 0x12)
					{
						if(bRepeat == 2)
							wSongNum = 0;		// play from begin of CD
							
						if(bRepeat || bRandom)
							goto RAND;		// start playing or random song
						else
						{
							event = EV_STOP;	// End of CD -> stop playing
							continue;
						}
					}
					
					if (bTimeMode)
						wPlayTime = wSongTime - wPlayTime;

					if (bTimeMode && wPlayTime >= 600)
						lcd_putchar('-');
					else
						lcd_putchar(' ');

					if (wPlayTime < 0) wPlayTime = 0;
					divt = udiv(wPlayTime, 600);
					*src++ = ((u08)divt.quot)?'0' + (u08)divt.quot % 10 : ((bTimeMode)?'-':' ');
					divt = udiv(divt.rem, 60);
					*src++ = '0' + (u08)divt.quot;
					*src++ = ':';
					divt = udiv(divt.rem, 10);
					*src++ = '0' + (u08)divt.quot;
					*src++ = '0' + (u08)divt.rem;			
					*src = 0;
				}
				else if (tot_sw < 12)	// 10-11
				{
					*dst++ = '#';
					src = pTrack;
					while ((*dst = *src++)) dst++;
					while (dst < &timestr[5])
						*dst++ = ' ';
					dst = timestr;
				}
				else if (tot_sw < 14)	// 12-13
				{
#ifdef DISPLAY_BITRATE
					if(bRandom || cdda)
#endif
						dst = randstr;
#ifdef DISPLAY_BITRATE
					else
					{
						dst = timestr;
						src = unsigned2str(wBitrate, timestr);
						while ((*dst = *src++)) dst++;
						*dst++ = 'k';
						while (dst < &timestr[5])
							*dst++ = ' ';
						dst = timestr;
					}
#endif
				}
				else if (tot_sw < 16)	// 14-15
				{
					dst = loudstr;
				}
				if (tot_sw > 9)
						lcd_putchar(' ');

				tot_sw += 1;
				
				if (tot_sw >= ((cdda)? 14 : 16))
					tot_sw = 0;

				lcd_puts(dst);
#if (LCD_LINE_LENGTH != 16)
				lcd_putchar(' ');		// separator
				lcd_putchar(1);			// speaker sign
#endif

#endif  //(LCD_LINES == 2)

//------------------------ TIME, TRACK NUMBER AND OTHER DISPLAYING   ( 4 LINES LCD ) --------------------------

#if (LCD_LINES == 4)
				lcd_gotoxy(lcd_steps, 2);			// Line 3
				src = timestr;

				if(!cdda)
				{
#ifdef OLD_VS1001
					wPlayTime = wPlayT / 10 + wAddTime;	// seconds
					if (wPlayTime > 32768) wAddTime = -(wPlayT / 10);
#else  //OLD_VS1001
					wPlayTime = vs1001_read(VS1001_PLAYTIME) + wAddTime;
#endif  //OLD_VS1001
				}
				else if (ATAPI_DA_GetSubC() == 0)
				{
					wTemp = ATAPI_DA_GetCurTrack();
					if(wTemp != wSongNum)
					{
						if(bRandom || bRepeat == 1)
							goto RAND;

						wSongNum = wTemp;
						pTrack = unsigned2str(wSongNum + 1, trackstr);
						wSongTime = GetSongTime((u08)wSongNum);
						dwFileSize = (u32)wSongTime << 8;
						goto PLAY_END;
					}
					else
					{
						wPlayTime = ATAPI_DA_GetCurTime(); 
						dwBytesLoaded = (u32)wPlayTime << 8;
					}
				}
				else if (ATAPI_CheckStatus() == CD_STAT_OPENED)	
				{
					Do_Eject();
					event = EV_IDLE;					// Media removed
					continue;
				}
				else if (ATAPI_DA_GetSubC() >= 0x12)
				{
					if(bRepeat == 2)
						wSongNum = 0;		// play from begin of CD
						
					if(bRepeat || bRandom)
						goto RAND;		// start playing or random song
					else
					{
						event = EV_STOP;	// End of CD -> stop playing
						continue;
					}
				}

				if (bTimeMode)
					wPlayTime = wSongTime - wPlayTime;

				if (bTimeMode && wPlayTime >= 600)
					lcd_putchar('-');
				else
					lcd_putchar(' ');

				if (wPlayTime < 0) wPlayTime = 0;
				divt = udiv(wPlayTime, 600);
				*src++ = ((u08)divt.quot)?'0' + (u08)divt.quot % 10 : ((bTimeMode)?'-':' ');
				divt = udiv(divt.rem, 60);
				*src++ = '0' + (u08)divt.quot;
				*src++ = ':';
				divt = udiv(divt.rem, 10);
				*src++ = '0' + (u08)divt.quot;
				*src++ = '0' + (u08)divt.rem;			
				*src = 0;
				lcd_puts(dst);				// display time, line 3, x= 15 - 19
				if(cdda)
					lcd_puts_p(PSTR("\nTrack "));
				else
					lcd_puts_p(PSTR("\n#"));
				lcd_puts(pTrack);
				lcd_putchar('/');
				dst = unsigned2str(wMaxSongs, timestr);
				lcd_puts(dst);
				lcd_puts_p(PSTR("      "));

				lcd_gotoxy(10+(LCD_LINE_LENGTH-13)*cdda,3);
#if (LCD_LINE_LENGTH != 16)
				if(bRandom)
					lcd_puts_p((bRandom == 1)?PSTR("RND "):PSTR("DIR "));  
				else
				{
#ifdef DISPLAY_BITRATE
					if(!cdda)
					{
						unsigned2str(wBitrate, timestr);
						lcd_puts(timestr+2);
						lcd_gotoxy(14,3);
					}
					else
#endif
						lcd_puts_p(PSTR("    "));
				}
				if (!cdda)
				{
					lcd_puts_p((bLoudness)?PSTR("LD"):PSTR("    "));
					if (bLoudness)
					{  
						lcd_putchar(ldchar(bLoudness));
						lcd_putchar(' ');
					}
					lcd_putchar(1);				// speaker sign
				}
#else //LCD_LINE_LENGTH == 16
				if(bRandom)
				{
					if(cdda)
						lcd_puts_p(PSTR("RND"));
					else
						lcd_puts_p((bRandom == 1)?PSTR("RA "):PSTR("RD "));  
				}
				else
					lcd_puts_p(PSTR("   "));

				if (!cdda)
				{
					lcd_puts_p((bLoudness)?PSTR("L"):PSTR("  "));  
					if (bLoudness)
						lcd_putchar(ldchar(bLoudness));
				}
#endif

				lcd_gotoxy(LCD_LINE_LENGTH-1,3);
#endif  //(LCD_LINES == 4)

//--------------------------------------------------------------------------------------------------------------------
				if(!cdda)
					lcd_putchar(0);		// Volume indicator on last position
				nDispUpd = 0;

#if ! ((F_VS1001 == 6144 + 0x8000) || (F_VS1001 == 12288) || defined(OLD_VS1001))
				if (vs1001_read(VS1001_CLOCKF) != F_VS1001)	// check for self reset of vs1001
				{
					vs1001_reset();				// soft resetting for right CLOCKF set
					Loud_Set(bLoudness);
				}
#endif
#ifdef ENABLE_SLEEP_TIMER
				if((u08)(nTimer-1) < 3 && (wTimerPresc % 100) < 6)
					TimerDisplay();
#endif
			} // if update time display
		} // if (bPlaying)
		else
		{									// If in STOP state
			if (nDispUpd > 5 )						// every 0.5 second
			{
				nDispUpd = 0;
				nWrites = ATAPI_CheckStatus();				// Checking CD drive status for:
				if ((nWrites != CD_STAT_OPENED && bTrayOpen) ||		// Manual Tray Close
				    (nWrites == CD_STAT_OPENED && !bTrayOpen))		// Manual Tray Open
				{
					function = 0;
					Do_Eject();					// Do open or close depend
				}							// value of bTrayOpen status
			}
		}
	} // while (true)

#endif //SETUP_REMOTE_CODES
}

//--------------------------------------------------------------------------------------------------------------------

// Auxillary functions definitions

u08 NO_CD[] __attribute__ ((progmem)) = "No Disc";
u08 WR_CD[] __attribute__ ((progmem)) = "Wrong Disc !";

bool LoadCD(void)					// true = ERROR
{
	u08 *songs;
	u08 stat;
	u08 retry = 6;
	u08 rt1;
	LCD_Info();
	if (bTrayOpen)
	{
		lcd_puts_p(PSTR("Close"));
		CD_Cmd(LOAD_CD);
		delayms(200);
		while(ATAPI_CheckStatus() == CD_STAT_OPENED);
	}
	bTrayOpen = false;
	nDispUpd = 0;
	wMaxSongs = 0;
	nLastEntry = 0;
	lcd_gotoxy(0,1);
	lcd_puts_p(PSTR("Finding CD..."));
RETRY:
	delayms(500);
	rt1 = 30;
	while(((stat = ATAPI_CheckStatus()) & 0x0F) == CD_STAT_NOT_READY)
	{
		delayms(500);
		if (stat == CD_STAT_OPENED)
		{
			Do_Eject();
			return true;
		}
		if(--rt1 == 0)
			goto NODISC;
	}

	if (stat == CD_STAT_NODISK)
	{
		if(--retry)
			goto RETRY;
NODISC:
		PRINT_p((u08*)&NO_CD[0]); EOL();
		LCD_Info();
		lcd_puts_p((u08*)&NO_CD[0]);
		return true;
	}

	if (stat == CD_STAT_OK)
	{
		SlowDown();
		lcd_gotoxy(0,LCD_LINES / 4);
		if (ISO_INIT() == 0)
		{
WRONG_DISC:	
			PRINT_p((u08*)&WR_CD[0]); EOL();
			LCD_Info();
			lcd_puts_p((u08*)&WR_CD[0]);
			CD_Cmd(STOP_CD);
			return true;
		}
		lcd_gotoxy(0,LCD_LINES / 4);
		PRINT_p(PSTR("Scanning..."));
		wMaxSongs = dirlist(DIRLIST_SCAN);
		if( !wMaxSongs )
			goto WRONG_DISC;

		songs = (unsigned2str(wMaxSongs , trackstr));  // jman number of songs in decimal
		PRINT(songs);
		PRINT_p(PSTR(" entries\n"));

		if (wMaxSongs == eeprom_read_word((u16*)EEPROM_MAX_SONGS))
			wSongNum = eeprom_read_word((u16*)EEPROM_SONG_NUMBER);
		else
			wSongNum = 0;

		eeprom_write_word((u16*)EEPROM_MAX_SONGS, wMaxSongs);
		InitRnd();

		if (!cdda)
		{
			lcd_gotoxy(0, LCD_LINES - 1);		// line 2 or 4
			lcd_puts(songs);
			lcd_puts_p(PSTR(" songs found."));
		}
		else
		{
			lcd_gotoxy(0, LCD_LINES / 4 + 1);	// line 2 or 3
			lcd_puts(songs);
			lcd_puts_p(PSTR(" tracks, "));		
			LCD_DiscTime();
		}

		CD_Cmd(STOP_CD);
		return false;
	}
	return true;
}

//--------------------------------------------------------------------------------------------------------------------

// converts 16 bit unsigned integer to string
// str[] should be 6 bytes at least. Function can return value != str
// and you have to use it as a output value start
u08* unsigned2str(u16 track, u08 *str)
{
	register u08 i;
	udiv_t divt;

	for (i = 0; i < 5; i++) *str++ = ' ';
	*str = '\0';
	divt.quot = track;
	do {
		divt = udiv(divt.quot, 10u);
		*(--str) = '0' + (u08)divt.rem;
	} while (divt.quot);
	return str;
}

u16 Numb2Int()				// Convert selected song number from string to integer
{
register u08 i;
u16 res = 0;
u16 p   = 1;
	for (i = pNum-number ; i>0 ;  i--)
	{
		res += p * (number[i-1]-'0');
		p *=10;
	}
	if (res > 999) res = 999;
	return res;
}


//--------------------------------------------------------------------------------------------------------------------

#ifdef OLD_VS1001
void BadHeadStrip(void)		// Special MP3 header patch to strip all invalid headers 
				// not needed for VS1001K chip
{
u16 j;
	for (j = 0; j < BUFFER_SIZE - 1; j++)
		if ((*(u16*)&BUFFER1P[j] & 0xfeff) != 0xfaff)
			BUFFER1P[j] = 0;
		else
			break;
	// End of special header patch
}
#endif

//--------------------------------------------------------------------------------------------------------------------

#ifdef ENABLE_SERIAL
u08 get_char_event(void)
{
	static u08 uart_chars[]="gGnpNPVvdDfFbratse";	//g-play, G-stop, n-next, p-prevous, N-next10, P-prev10,
	register u08 i;					//v-volup, V-voldown, d-nextdir, D-prevdir, f-fwd, F-rew,
	register u08 code = UART_ReceiveByte();		//b-loudness, r-random, a-repeat, t-time, s-sleep, e-eject

	for (i=0; i< sizeof(uart_chars); i++)
		if (code == uart_chars[i])
			return (i+1);
	return EV_IDLE;
}
#endif // ENABLE_SERIAL

//--------------------------------------------------------------------------------------------------------------------

#ifdef ENABLE_IR
// get a remote control event

u16 TableIR[] = {IR_PLAY, IR_STOP, IR_NEXT, IR_PREV, IR_NEXT10, IR_PREV10, IR_VOLUP, IR_VOLDN, IR_NEXTD, IR_PREVD,
		 IR_FFWD, IR_FREW, IR_LOUDNESS, IR_RANDOM, IR_REPEAT, IR_REMAIN, IR_SLEEP, IR_EJECT, IR_FUNCTION};

 u16 NumTab[] = {IR_0, IR_1, IR_2, IR_3, IR_4, IR_5, IR_6, IR_7, IR_8, IR_9};

u08 ir_event(void)
{
	static u16 Lcod;
	u08 i;
	u16 code = get_rec80();
	if (Lcod == code && nKeyTime <= DOUBLE_TRAP)		// Trap for some IR transmitters that fast repeat
		return EV_IDLE;					// code of pressed key

	for (i = 0; i < (sizeof TableIR / sizeof TableIR[0]); i++)
		if (TableIR[i] == code)
		{
			LastEvent = i+1;				// Save last event
RET_LAST:		nKeyTime = 0;
			Lcod = code;
			return LastEvent;
		}

	if (code == IR_RESET)
	{
		wdt_enable(0x01);
		while(true);					// Reset player by watchdog
	}

#ifdef ENABLE_AUTOREPEAT
	if ((nKeyTime >= 2) && (LastEvent == EV_VOLUP || LastEvent == EV_VOLDN || 
				LastEvent == EV_FFWD  || LastEvent == EV_FREW))		// Autorepeat feature on some keys
		goto RET_LAST;
#endif

	for (i=0; i<10; i++)
		if (NumTab[i] == code)			// Check for numeric keypress code
		{
			*pNum++ = i+'0';		// Add numeric key ascii character to string
			*pNum = 0;			// String end flag
			LastEvent = EV_NUMBER;
			goto RET_LAST;
		}
			
	return EV_IDLE;
}

#else
 #define ir_event() EV_IDLE
#endif // ENABLE_IR

//--------------------------------------------------------------------------------------------------------------------

// get a keyboard event - edhanced by MIS

 u08 TableKbdShort[] = KBD_SHORT;	// Key definitions moved to "Constants.h" file !!!
 u08 TableKbdLong[] = KBD_LONG;		// Key definitions moved to "Constants.h" file !!!

u08 get_key_event(void)
{
	u08 event = EV_IDLE;
	static u08 keydown = 0;
	register u08 i;
	MCUCR &= ~ _BV(SRE);	// disable ExtRAM
	DDRA = 0xff;			// set port as output
	PORTA = 0;			// set all pins to 0
	asm volatile("nop");		// allow time for port to activate

	if (keydown)			// if a key was pressed earlier
	{
		// see if it has been released
		if (bit_is_set(PIND, PD3))				// check bit
		{
			// no key is down, return last key seen
			if (nKeyTime < 8 && nKeyTime > 0)		// max 0.8 sec for shortpress
				event = TableKbdShort[keydown];		// short keypress event
			keydown = 0;
		}
		else
			if ((nKeyTime & 0x7F) >= 8)			// long keypress ? 
			{
				event = TableKbdLong[keydown];		// long keypress event
				if (event == EV_VOLUP || event == EV_VOLDN || 
				    event == EV_FFWD  || event == EV_FREW)
					nKeyTime = 0x86;		// 0.2 sec repeat time for volume and fast forward
				else
					nKeyTime = 0x82;		// 0.6 sec repeat time
			}	


	}
	else
	{	
		if (bit_is_clear(PIND, PD3))	// check if a key is down
		{
			// a key is active, check which one
			for (i = 0; i < 8; i++)
			{
				PORTA = ~(1 << i);			// write bit mask
				asm volatile("nop");			// wait a while
				asm volatile("nop");			// wait a while
				if (bit_is_clear(PIND, PD3)) 	// this bit ?
				{
					keydown = i + 1;		// store the key
					nKeyTime = 0;			// clear keypress timer
					break;				// and exit
				}
			}
		}
	}

	MCUCR |= _BV(SRE);					// enable ExtRAM
	return event;
}

//--------------------------------------------------------------------------------------------------------------------

void send_sinewave_beeps(void)
{
#ifdef PWR_BEEPS
	static u08 buf[] = { 0x53, 0xEF, 0x6E , 0x30, 0x45, 0x78, 0x69, 0x74 };
	register u08 i, j = 3;

	while (j-- != 0)
	{
		for (i = 0; i < 4; i++)
			vs1001_send_data(buf[i]);
		vs1001_nulls(4);
		delayms(100);		// delay 100ms

		for (/*i = 4*/; i < 8; i++)
			vs1001_send_data(buf[i]);
		vs1001_nulls(4);
		delayms(100);
	}	
#endif
}

//--------------------------------------------------------------------------------------------------------------------

void defchar(u08 dat)
{
	register u08 i=5;
	lcd_data(0x1f);
	while (i--)
		lcd_data(dat);
	lcd_data(0x1f);
	lcd_data(0x00);				// cursor line
}

void setvolume(u08 v)
{
	vs1001_setvolume(v, v);
	eeprom_write_byte((u08*)EEPROM_VOLUME, v);
	register u08 i;
	lcd_command(0x40);
	for (i = 0; i < 8; i++)
	{
		if (v <= i * (VOL_STEP*(NUM_VOL_STEPS/8)))
#if (LCD_LINE_LENGTH == 16)
			lcd_data((1 << ((10 - i) / 2)) - 1);	// Volume idicator mod
#else
			lcd_data(0x1f << ((i+1)*5 / 10));	// Volume idicator mod (reversed)
#endif
		else
			lcd_data(0);
	}

#if (LCD_LINE_LENGTH != 16)
	static u08 vsc[] = {0x03, 0x05, 0x19, 0x19, 0x19, 0x05, 0x03, 0x00};	// speaker sign
	for (i=0; i < 8; i++)
		lcd_data(vsc[i]);				// load to CG_RAM (char 1)
#endif
	lcd_command(0x80);
}

void lcd_frame(void)
{
	register u08 i = lcd_steps - 2;
	lcd_command(0x50);			// start definition of CG-RAM from char code 2
	defchar(0x10);				// start bar (2)
	defchar(0x1f);				// full bar (3)
	defchar(0x00);				// empty bar (4)
	defchar(0x01);				// ended bar (5)
	lcd_gotoxy(0, LCD_LINES / 4 + 1);
	lcd_putchar(2);	
	while (i--)
		lcd_putchar(4);	
	lcd_putchar(5);	
}

u08 barcalc(u08 dummy)				// assembler optimized function to calculate progressbar position
{
// (u08)(LCD_STEPS * (dwBytesLoaded>>8) / (dwFileSize>>8));

	asm volatile ("mov r18, r24");
	asm volatile ("clr r19");
	asm volatile ("clr r20");
	asm volatile ("clr r21");
	asm volatile ("lds r22,(dwBytesLoaded)+1");
	asm volatile ("lds r23,(dwBytesLoaded)+2");
	asm volatile ("lds r24,(dwBytesLoaded)+3");
	asm volatile ("clr r25");
	asm volatile ("call __mulsi3");
	asm volatile ("lds r18,(dwFileSize)+1");
	asm volatile ("lds r19,(dwFileSize)+2");
	asm volatile ("lds r20,(dwFileSize)+3");
	asm volatile ("clr r21");
	asm volatile ("call __udivmodsi4");
	asm volatile ("mov r24,r18");
	return dummy;
}

void dispbar(void)
{
	register u08 i, a, nSteps;

	nSteps = barcalc(lcd_steps * 5 -1);
	a = nSteps / 5;						// # of full bars
	static u08 data[] = { 0x10, 0x18, 0x1C, 0x1E, 0x1F };

	lcd_command(0x50);
	defchar(data[nSteps % 5] | (a == lcd_steps - 1));

	lcd_gotoxy(0,LCD_LINES / 4 + 1);

	for(i=0; i<a; i++)
		lcd_putchar(3);					// print full bars

	if(i<lcd_steps)
	{
		lcd_putchar(2);
		i++;
		for (; i<lcd_steps - 1; i++)
			lcd_putchar(4);	
		if (i<lcd_steps)
			lcd_putchar(5);	
	}
}

//--------------------------------------------------------------------------------------------------------------------

bool RetrieveMP3Data(void)
{
	u08 nRes;
	if (bBuf1Empty)
	{

		nRes = CheckAndAdvanceBlock();
		if (nRes == NO_DATA)
			return false;
		if (nRes == DATA_EXIST)
		return (ReadBuffer(BUF1));		// Was disk operation, need pump in
	}

	if (bBuf2Empty)
	{
		nRes = CheckAndAdvanceBlock();
		if (nRes == NO_DATA)
			return false;
		if (nRes == DATA_EXIST)
		return (ReadBuffer(BUF2));		// Was disk operation, need pump in
	}
	return false;
}

