/* **********************************************************************
**
**  Copyright (C) 2003  Jesper Hansen <jesperh@telia.com> and 
**			Romuald Bialy (MIS) <romek_b@o2.pl>.
**
*************************************************************************
**
**   This file is part of the yampp system.
**
**  This program is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License
**  as published by the Free Software Foundation; either version 2
**  of the License, or (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software Foundation, 
**  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
*********************************************************************** */

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/pgmspace.h>

#include "uart.h"

#ifdef ENABLE_SERIAL


// UART global variables
volatile bool UART_CharReceived = false;
volatile bool UART_RxBufOverflow = false;
volatile u08  UART_RxChar = 0;

// ---- UART core functions --------------------------------------------

// Initialisation
void UART_Init(void)
{
	UART_CharReceived	= false;
	UART_RxBufOverflow	= false;

	// enable RxD/TxD and interrupts

	#define UART_FLAGS (_BV(RXCIE) | _BV(RXEN) | _BV(TXEN))
	UCSR0B = UART_FLAGS;

	UBRRH = 0;
	UBRR0 = (u08)UART_BAUD_SELECT;			// set baud rate
	sei();						// enable interrupts
}


// Receiving part

// UART Receive Complete Interrupt Function

#if defined (__AVR_ATmega162__)
 SIGNAL(SIG_USART0_RECV)
#else
 SIGNAL(SIG_UART0_RECV)
#endif
{
	if (UART_CharReceived)			// buffer overflow
		UART_RxBufOverflow = true;
    	UART_RxChar = UDR0;			// Store received character
    	UART_CharReceived = true;		// Now, and only now we can indicate
						//  that the UART has received a character
}


// Retrieve received character or wait until receive
u08 UART_ReceiveByte(void)
{
	register u08 temp;
    	while(!UART_CharReceived);		// wait for UART indicates that a character
						//  has been received. Watchdog protected
    	temp = UART_RxChar;			// first we have to take received character
	UART_CharReceived = false;		// and only then report that buffer is free
    	return temp;				// return received character
}


// Checks and clear Rx buffer overflow bug
bool UART_CheckRxBuf(void)
{
	register bool temp;
	temp = UART_RxBufOverflow;
	UART_RxBufOverflow = false;
	return temp;
}


// Transmit part

void UART_SendByte(u08 Data)
{
	// wait for UART to become available
	while ((UCSR0A & _BV(UDRE)) != _BV(UDRE));

	// send character
 	UDR0 = Data;
}


void UART_PrintfEndOfLine(void)
{
	UART_SendByte('\r');
	UART_SendByte('\n');
}


static void UART_PrintfU4(u08 Data)
{
//	 Send 4-bit hex value 
	u08 Character = Data & 0x0f;
	Character += '0';
	if (Character > '9')
		Character += 'A' - '0' - 10;
	UART_SendByte(Character);
}


void UART_Printfu08(u08 Data)
{
//     Send 8-bit hex value 
    UART_PrintfU4(Data >> 4);
    UART_PrintfU4(Data);
}


void UART_Printfu16(u16 Data)
{
//     Send 16-bit hex value 
    UART_Printfu08(Data >> 8);
    UART_Printfu08(Data);
}


void UART_Printfu32(u32 Data)
{
//    Send 32-bit hex value 
    UART_Printfu16(Data >> 16);
    UART_Printfu16(Data);
}


//------------------------------------------------------------------------------------

void UART_Puts(u08* pBuf)
{
	register u08 c;
	while ((c = *pBuf++))
	{
		if (c == '\n')
			UART_SendByte('\r');		// for stupid terminal program
		UART_SendByte(c);
	}
}

void UART_Puts_p(char const *p)
{
	register u08 c;
	while ((c = pgm_read_byte(p++)))
	{
		if (c == '\n')
			UART_SendByte('\r');		// for stupid terminal program
		UART_SendByte(c);
	}
}


void UART_Putsln(u08* pBuf)
{
	UART_Puts(pBuf);
	UART_PrintfEndOfLine();
}


#endif // ENABLE_SERIAL
