/* **********************************************************************
**
**  Copyright (C) 2003  Jesper Hansen <jesperh@telia.com> and 
**			Romuald Bialy (MIS) <romek_b@o2.pl>.
**
*************************************************************************
**
**   This file is part of the yampp system.
**
**  This program is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License
**  as published by the Free Software Foundation; either version 2
**  of the License, or (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software Foundation, 
**  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
*********************************************************************** */

#include <avr/io.h>
#include <avr/pgmspace.h>

#include "lcd.h"
#include "mem.h"
#include "delay.h"
#define enable_ram()	 MCUCR |= _BV(SRE)

static u08 lcd_x, lcd_y;

/*************************************************************/
/********************** LOCAL FUNCTIONS **********************/
/*************************************************************/

static void LCDSetAddress(u08 adr) 
{ 
	MCUCR &= ~ _BV(SRE);				// disable ExtRAM
	DDRA = 0xff;						// set port as output
	PORTA = adr;						// write addres
	PORTE |= _BV(PE1);
	PORTE &=~ _BV(PE1);				// flip ALE
}


static void lcd_waitbusy(void)				// loops while lcd is busy 
{							// no watchdog in this function
	register u08 i = 0;				// should work fast enough 
	register u08 data;

	// setup RS and RW pins
   	LCDSetAddress(LCD_IO_FUNCTION | LCD_IO_READ);
	LCD_DATA_DDR = 0;				// set port as input
	LCD_DATA_PORT = 0xff;				// enable pullup
	do
	{
		lcd_e_high();				// set LCD enable high
		data = LCD_DATA_PIN;		// read byte
		lcd_e_low();				// set LCD enable low
		i++;
	} while ((data & (1 << LCD_BUSY)) && (i != 255));
	enable_ram();
}


static void lcd_write(u08 data, u08 rs) 
{
	lcd_waitbusy();
	// setup RS and RW pins
   	LCDSetAddress(rs | LCD_IO_WRITE);
	LCD_DATA_DDR = 0xff;				// set port as output
	LCD_DATA_PORT = data;				// write byte
	lcd_e_high();						// set LCD enable high
	lcd_e_low();						// set LCD enable low
	enable_ram();
}


static void lcd_newline(void)		// goto start of next line
{
    lcd_x = 0;
    if (lcd_y < LCD_LINES)
        lcd_y++;
}


void lcd_goto(void) 				// goto position (lcd_x,lcd_y)
{
#if (LCD_LINE_LENGTH == 16)
 static	u08 lcd_line[] = {0x00, 0x40, 0x10, 0x50};
#else
 static	u08 lcd_line[] = {0x00, 0x40, 0x14, 0x54};
#endif
	lcd_command((1 << LCD_DDRAM) + lcd_line[lcd_y] + lcd_x);
}



/*************************************************************/
/********************* PUBLIC FUNCTIONS **********************/
/*************************************************************/


void lcd_command(u08 cmd)			// send commando <cmd> to LCD 
{
	lcd_write(cmd, LCD_IO_FUNCTION);
}


void lcd_data(u08 data)				// send data <data> to LCD
{
	lcd_write(data, LCD_IO_DATA);
}


void lcd_gotoxy(u08 x, u08 y)			// goto position (x,y) 
{
	lcd_x = x;  lcd_y = y;
	lcd_goto();
}


void lcd_clrscr(void)					// clear lcd
{
	lcd_x = lcd_y = 0;
	lcd_command(1 << LCD_CLR);
	delayms(3);					// 3 ms delay
}


void lcd_putchar(u08 data)				// print character to
{								//  current cursor position
	if (data == '\n')
	{
		lcd_newline();
		lcd_goto();
	}
	else
	{
		if (lcd_x < LCD_LINE_LENGTH)
		{
			lcd_x++;
			lcd_write(data, LCD_IO_DATA);
		}
	}
}


void lcd_puts(u08* str)				// print string on lcd 
{								//  (no auto linefeed)
	while (*str)
		lcd_putchar(*str++);
}

void lcd_puts_p(char const *p)			// print string from flash on lcd 
{	
	register u08 b;
		while ((b = pgm_read_byte(p++)))
			lcd_putchar(b);
}



// cursor:   0 = off, 2 = on, 3 = blinking
// fnc: see LCD_FUNCTION_xxx
void lcd_init(u08 cursor, u08 fnc)
{
	register u08 i;

	fnc |= (1 << LCD_FUNCTION);

	for (i = 0; i < 4; i++)					// reset lcd
	{
		delayms(20);					// delay
		lcd_write(fnc, LCD_IO_FUNCTION);	// reset function
	}

	lcd_command(1 << LCD_ON);
	lcd_clrscr();
	lcd_command(LCD_MODE_DEFAULT);
	lcd_command((1 << LCD_ON)|(1 << LCD_ON_DISPLAY) | cursor);
} 

