/* **********************************************************************
**
**  Copyright (C) 2003  Jesper Hansen <jesperh@telia.com> and 
**			Romuald Bialy (MIS) <romek_b@o2.pl>.
**
*************************************************************************
**
**   This file is part of the yampp system.
**
**  This program is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License
**  as published by the Free Software Foundation; either version 2
**  of the License, or (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software Foundation, 
**  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
*********************************************************************** */

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <string.h>

#include "mem.h"
#include "uart.h"
#include "atapi.h"
#include "iso9660.h"
#include "lcd.h"
#include "delay.h"

//Global variables

bool joliet;
bool cdda;
u32  dwFileSize;		// Current file size
u32  dwBytesLoaded;		// Counter of bytes loaded from CD
bool bBuf1Empty	= true;		// Empty flags for buffers
bool bBuf2Empty	= true;
u08  nLastPosD = 0;		// Directory tabele last element number
u08  nLastEntry = 0;		// For directory change checking
u32  dwCurrentBlock;		// Current being read block

u32 SessionOffset;
static u32 PathBlock;		// Block number for Path Tabele
static u32 dwDirInCache;
static u08 PathSize;		// Size of Path Tabele in blocks
static u08 *dst;		// Temp var, do not need to be in stack
static u16 wCurEntry;		// Couner of current dir entry during get_de() 					
static u08 dirlisting = 0;	// Mode of diectory browsing 
static u16 wNewEntry;
bool FindBootBlock(u32 sig);

extern bool bError;

//============================================================================================
// National characters decoding and replacing
//============================================================================================

charspair TableChars[] = NATIONAL_DEF;
u08 replacechar(u16 data)		// national chars replace
{
	u08 i;
	if (joliet)
	{
		for(i=0; i< sizeof(TableChars) / sizeof(TableChars[0]); i++)
			if (TableChars[i].natJol == data)
				return TableChars[i].norm;
	}
	else
		data >>= 8;

	return ((u08)data);
}

extern void printsp(void);

//============================================================================================
// Get inserted CD toc and name
//============================================================================================

u08 ISO_INIT(void)
{
	u08 i;

	*(u16*)(LONG_NAME - 2) = 0xFFFF;			// This are special 3 bytes, 
	*(LONG_NAME - 3) = 0xFF;				//  placed to help in .mp3
 								//  extension checking
	*CD_NAME = 0;
	cdda = false;

	if (ATAPI_DA_GetToc() == 0)				// CD-DA detect
		goto CDDA;

	joliet = true;

	SessionOffset = ATAPI_GetIsoToc();

	i = FindBootBlock(0x30444302);
	if (i==2)
		goto CDDA;

	if (i==0)						// Check secondary Boot Block
	{
		joliet = false;					// Possible ISO disc
		if (FindBootBlock(0x30444301) != 1) 		// Check primary Boot Block
			return 0;				// RESULT 0 = ERROR, NO VALID CD
	}
	PORTD &= ~ _BV(SWITCH_PIN);					// switch to VS1001 output
	dst = CD_NAME;
	for( i=0 ; i < ISO_VOL_ID_LEN; i += joliet+1 )
	{
		*dst = 	replacechar((BUFFER1P[ISO_VOL_ID + i]<<8) + BUFFER1P[ISO_VOL_ID + i + 1]);
		if(*(dst++) == 0) break;
	}
	*dst = 0;
	lcd_puts(CD_NAME);
	printsp();
	lcd_putchar('\n');
	PRINT_p(PSTR("Found "));
	if (joliet)
	{
		lcd_puts_p(PSTR("Joliet disc."));
		PRINT_p(PSTR("Joliet disc: ")); 
	}
	else
	{
		lcd_puts_p(PSTR("ISO9660 disc."));
		PRINT_p(PSTR("ISO9660 disc: "));
	}
	PRINT(CD_NAME); EOL();
	PathBlock = *(u32*)(BUFFER1P + ISO_PTABLE);			// Location of Path Tabele
	PathSize  = (*(u32*)(BUFFER1P + ISO_PTABLE - 8)) / BPS +1;	// Size of Path Tabele
	if (PathSize>2) PathSize=2;							// no more ram to load bigger PT
	return 1;											// RESULT 1 = DATA CD

CDDA:
	cdda = true;
	strcat(CD_NAME,"Audio CD");
	lcd_puts(CD_NAME);
	printsp();
	PRINT(CD_NAME); EOL();
	PORTD |= _BV(SWITCH_PIN);			// switch to CD-ROM audio output
	return 2;							// RESULT 2 = AUDIO CD
}	

u08 FindBootBlock(u32 sig)
{
	u08 i;
	for(i=0; i<5; i++)
	{
		while(ATAPI_ReadLBA(SessionOffset + ISO_BOOT_BLOCK + i , BUFFER1P))
		{
			if (ATAPI_DA_GetToc() == 0)			// try CD-DA detect
				return 2;
		}

		if(*(u32*)BUFFER1P == sig)
			return 1;
	}
	return 0;
}



//============================================================================================
// Search for files inside the directory
//============================================================================================
bool GetFileList(u32 adr)
{
	register u08 i,n = 0;
	u16 offset = 0;
	struct direntry *de;
	bool block_rd = false;

	if (adr != dwDirInCache)
	{
		dwDirInCache = adr;					// store LBA address
		block_rd = true;
		do{
			while(ATAPI_ReadLBA(adr, DIR_CACHE + offset));	// load sector
			de = (struct direntry *)(DIR_CACHE + offset);	// prepare structure data

			if (offset && (de->Flags == 2))			// if read not first Dir Block and it start
				break;					// with directory entry, this mean that is
									// next directory, not old continuation
			adr++;
			offset += 0x0800;
		}while((de->LEN_DR > 0) && ((u08)de->FileLocL == 0) && (offset < 0x4000));
									// check for valid entry at sector begin
									// and if found, load next sector
		if (offset < 0x4000)
			memset(DIR_CACHE + offset, 0, 0x0800);		// clear next sector of DirCache memory

		adr = dwDirInCache; 					// restore address
	}
CONTINUE:
	offset = n * 0x0800;						// calculate offset from sector number
	do
	{
		*LONG_NAME = 0;
		de = (struct direntry *)(DIR_CACHE + offset);
		if ((de->Flags & (Flag_NotExist | Flag_Directory)) == 0)	// Normal file
		{
			dst = LONG_NAME;
			for(i=0; i < de->Len_Fi; i += (joliet+1))
			{
				*dst = replacechar((de->FileName[i]<<8) + de->FileName[i+1]);
				if (*dst != ';')
					dst++;
				else
					break;
			}

			*dst = 0;	
			if (*--dst == '3' && (*--dst == 'p'  ||  *dst == 'P')  &&	// if MP3 file
			   (*--dst == 'm'  ||  *dst == 'M') &&	*--dst == '.')

			{
				*dst = 0;					// cut .MP3 extension
				if (dirlisting == DIRLIST_VERBOSE)	
				{						// print the name
					UART_Puts(DIR_NAME);
					PRINT_p(PSTR(" / "));
					UART_Puts(LONG_NAME);
					EOL();
				}
				// If this is the entry we're looking for, exit
				if (wCurEntry == wNewEntry)
				{
					dwCurrentBlock = de->FileLocM;
					dwFileSize = de->FileLenM;
					return true;
				}
				if(block_rd && dirlisting && (nLastPosD < MAX_DIRS))	
										// This is first mp3 file in directory
				{
					block_rd = false;
					DirsTable[nLastPosD].song_num = wCurEntry; 
										// first file in dir for dir navigation
					DirsTable[nLastPosD].dir_block = adr; 
										// dir Block number for fast load
					nLastPosD++;
					//UART_Printfu08(wCurEntry); UART_SendByte(' '); UART_Printfu32(adr); EOL();  // Debug
				}
				wCurEntry++;				// Increase song number

			} // if MP3 file
		} // if Normal file
		offset += de->LEN_DR;

	} while ((de->LEN_DR > 0) && ((u08)de->FileLocL == 0));		// Check for an empty or not directory record

	if (((offset % 0x0800) > 1900) && (++n < 8))			// Continue possible in next block
		goto CONTINUE;						// continue searchching

	return false;							// Reached end of directory, track not found
}

//============================================================================================
// Read Path Tabele and call to GetFileList on all found directory
//============================================================================================
bool get_de(void)

{
	register u08 i;
	u16 offset = 0;
	struct pathtab *pt = 0;
	while(ATAPI_ReadLBA(PathBlock, BUFFER1P));				// Read Path Tabele to Buffer1
	if (PathSize == 2)							// if more than 1 block
		while(ATAPI_ReadLBA(PathBlock + 1, BUFFER1P + 2048));		// Read rest of PT
	while(1)
	{
		*DIR_NAME = 0;
		pt = (struct pathtab *)(BUFFER1P + offset);			// Adres of current PT record
		if (pt->DirNameLen > 0)
		{
			dst = DIR_NAME;
			for(i=0; i < pt->DirNameLen; i += (joliet+1))		// Get Dir name
				*dst++ = replacechar((pt->DirName[i]<<8) + pt->DirName[i+1]);
			*dst = 0;
#if (LCD_LINES == 2)
			if(*DIR_NAME)
				strcat(DIR_NAME," / ");
#else
#ifndef ALTERNATE_SCROLL
			if(strlen(DIR_NAME) > LCD_LINE_LENGTH)
				strcat(DIR_NAME," >> ");
#endif
#endif
			if (GetFileList(pt->DirLoc))				// Get File list inside directory
				return true;					// Return if found entry
			offset += (8 + pt->DirNameLen);				// Calculate next PT record offset
			if (pt->DirNameLen % 2  == 1) 				// Skip padding field if DirNameLen
				offset++;					// is an odd number
		}
		else
			return false;	
	}
}

//============================================================================================
// Initial read CD contents
//============================================================================================

u16 dirlist(u08 mode)
{
	dirlisting = mode;
	nLastPosD = 0;			// Directory tabele last element number = 0
	wCurEntry = 0;			// current entry number = 0
	wNewEntry = -1;
	if (cdda)
		wCurEntry = ATAPI_DA_GetTrackN();
	else
		get_de();		// scan through all
	dirlisting = 0;
	nLastEntry = -1;
	return wCurEntry;
}


//============================================================================================
// This function sets information about MP3 file number 'entry'
// Returns true in file was found and false otherwise
//============================================================================================
bool get_dir_entry(u16 entry)
{
	u08 i = nLastPosD-1;
	wNewEntry = entry;

	while (i > 0 && DirsTable[i].song_num > entry) i--;

	if (nLastEntry == i)					// check for dir change
	{							// song in the same dir
		wCurEntry = DirsTable[i].song_num;		// set current entry at first file in dir
#if (LCD_LINES == 2)
		dst = strchr(DIR_NAME,'/');			// check for slash in directory name
		if(dst)						// if slash found
			*(dst+2) = 0;				// cut file name from dir string
		else							// slash not found, it's root directory
			*DIR_NAME = 0;				// clear old name
#endif
		return GetFileList(DirsTable[i].dir_block);	// get directory from current dir block
	}
	else	
	{							// song in different dir
		wCurEntry = 0;
		nLastEntry = i;					// save actual dir number for directory change check
		return get_de();				// search from root directory
	}
}


//============================================================================================
// Check for more avaliable data to load, or error occurence
//============================================================================================
bool CheckAndAdvanceBlock(void)
{
	return ((dwBytesLoaded <= dwFileSize - BUFFER_SIZE) && (!bError));
}

//============================================================================================
// Read block from CD and advance blocks counter
//============================================================================================

u08 rdblock(u08* pBuf)
{
	u08 retry = 5;
	while((bError = ATAPI_ReadLBA(dwCurrentBlock, pBuf)) && (--retry == 0));
	dwCurrentBlock++;
	dwBytesLoaded += BPS;
	return bError;
}

//============================================================================================
// Read two block from CD to one of buffer BUF1 or BUF2
//============================================================================================
bool ReadBuffer(u08 nBuf)
{
	u08* pBuf = (nBuf == BUF1) ? BUFFER1P : BUFFER2P;

	if(rdblock(pBuf))
		return bError;

	if(rdblock(pBuf+BPS))
		return bError;

	if (nBuf == BUF1)
		bBuf1Empty = false;
	else
		bBuf2Empty = false;

	return 0;
}
