/* **********************************************************************
**
**  Copyright (C) 2003  Jesper Hansen <jesperh@telia.com> and 
**			Romuald Bialy (MIS) <romek_b@o2.pl>.
**
*************************************************************************
**
**   This file is part of the yampp system.
**
**  This program is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License
**  as published by the Free Software Foundation; either version 2
**  of the License, or (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software Foundation, 
**  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
*********************************************************************** */

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <string.h>

#include "atapi.h"
#include "delay.h"
#include "mem.h"
#include "uart.h"

static void SetAddress(u08 addr);
static u08  ReadBYTE(u08 addr);
static void WriteBYTE(u08 addr, u08 dat);
static u08  CheckBusy(void);
static u08  CheckDataReq(void);
static u08  GetDataFromCD(u08 *buffer);

extern bool bPlaying;

#define MINIBUFFERSIZE 128
u08 minibuffer[MINIBUFFERSIZE];

#define enable_ram()	 MCUCR |= _BV(SRE)

/*
void regdump(void)
{
	u08 i;
	for(i=0;i<8;i++)
	{
		UART_Printfu08(ReadBYTE(CMD_A | i)); 
		UART_SendByte(32);
	}
	PRINT("\r\n");
}
*/
//----------------------------------------------------------------------------
// ATAPI Reset device
//----------------------------------------------------------------------------
void ATA_SW_Reset(void)
{
	u08 i=10;
	WriteBYTE(CMD_A | CMD_COMMAND, ATAPI_RESET);	// ATAPI Soft Reset command
	delayms(500);
	while(CheckBusy() && i)
		i--;
}

//----------------------------------------------------------------------------
// Init ATA
//----------------------------------------------------------------------------

u08 set_drive(u08 drive)
{
  	WriteBYTE(CMD_A | CMD_DEV_HEAD, drive);	// Set Drive/Head register
	delay10us();
	ATA_SW_Reset();						// Reset drive

	if( ReadBYTE(CMD_A | CMD_SECT_CNT) == 1 && ReadBYTE(CMD_A | CMD_SECT_NUM) == 1 &&		// check ATAPI signature
	    ReadBYTE(CMD_A | CMD_CYL_LOW) == 0x14 && ReadBYTE(CMD_A | CMD_CYL_HIGH) == 0xEB )
		return(1);
	else
		return(0);
}


u08 ATA_Init(void)
{
	if( set_drive(DH_CHS0) )	// try Drive 0 (MASTER)
		return(0);
	if( set_drive(DH_CHS1) )	// try Drive 1 (SLAVE)
		return(0);
	return(1);
}

//----------------------------------------------------------------------------
// Clear Packet command buffer;
//----------------------------------------------------------------------------
void ATAPI_ClearBuf(void)
{
	memset(minibuffer, 0, 12);
}

// ===========================================================================

// ***************************************************************************
// 				AUDIO CD FUNCTIONS
// ***************************************************************************

// helper functions

u08 calcoffset(u08 track, u08 offset)
{
	return DIR_CACHE[(track * 8) + offset];
}

u16 convert_to_ms(u08 m, u08 s)
{
	return( (u16)m*60 + (u16)s );
}

//----------------------------------------------------------------------------
// Init Audio, get TOC
//----------------------------------------------------------------------------
u08 ATAPI_DA_GetToc(void)
{
	u08 err;
	DIR_CACHE[5] = 0xFF;				// clear audio toc marker
	ATAPI_ClearBuf();				// clear command buffer
	minibuffer[0]=0x43;				// READ TOC command
	minibuffer[1]=0x02;				// MSF Format
	minibuffer[7]=0x08;				// 2 kB
	err = GetDataFromCD(DIR_CACHE);			// Get data from CD

	if ((DIR_CACHE[5] & 0x0f) > 3)			// if not audio CD
		err = CD_STAT_NOT_READY;		// return no audio CD

	return (err);
}

//----------------------------------------------------------------------------
// get M for Track #
//----------------------------------------------------------------------------
u08  ATAPI_DA_GetM(u08 Track)
{
	return(calcoffset(Track, 9));
}

//----------------------------------------------------------------------------
// get S for Track #
//----------------------------------------------------------------------------
u08  ATAPI_DA_GetS(u08 Track)
{
	return(calcoffset(Track, 10));
}

//----------------------------------------------------------------------------
// get F for Track #
//----------------------------------------------------------------------------
u08  ATAPI_DA_GetF(u08 Track)
{
	return(calcoffset(Track, 11));
}

//----------------------------------------------------------------------------
// get total # of tracks on the CD
//----------------------------------------------------------------------------
u08  ATAPI_DA_GetTrackN(void)
{
	union u16convert num;
	num.bytes.high = DIR_CACHE[0];
	num.bytes.low = DIR_CACHE[1];
	return( (num.value - 4) / 8 );
}

//----------------------------------------------------------------------------
// get total disc time (in seconds)
//----------------------------------------------------------------------------
u16  ATAPI_DA_GetDiscTime(void)
{
	return(convert_to_ms( ATAPI_DA_GetM(ATAPI_DA_GetTrackN()), ATAPI_DA_GetS(ATAPI_DA_GetTrackN()) ) -2 );
}

//----------------------------------------------------------------------------
// Play From M:S:F
//----------------------------------------------------------------------------
u08  ATAPI_DA_PlayMSF(u08 m, u08 s, u08 f)
{
	ATAPI_ClearBuf();
	minibuffer[0] = 0x47;					// Command code (PLAY)
	minibuffer[3] = m;					// start M
	minibuffer[4] = s;					// start S
	minibuffer[5] = f;					// start F
	minibuffer[6] = ATAPI_DA_GetM(ATAPI_DA_GetTrackN());	// end M
	minibuffer[7] = ATAPI_DA_GetS(ATAPI_DA_GetTrackN());	// end S
	minibuffer[8] = ATAPI_DA_GetF(ATAPI_DA_GetTrackN());	// end F
	return ATAPI_Command();					// 0 if OK, else ERROR
}


//----------------------------------------------------------------------------
// Play FromTrack
//----------------------------------------------------------------------------
u08  ATAPI_DA_Play(u08 Track)
{
	return ATAPI_DA_PlayMSF(ATAPI_DA_GetM(Track), ATAPI_DA_GetS(Track), 
				ATAPI_DA_GetF(Track));
}

//----------------------------------------------------------------------------
// Stop Playing
//----------------------------------------------------------------------------
u08  ATAPI_DA_Stop(void)
{
	ATAPI_ClearBuf();
	minibuffer[0] = 0x4E;			// Command code (STOP)
	return ATAPI_Command();			// 0 if OK, else ERROR
}

//----------------------------------------------------------------------------
// Pause, resume (bResume == true) playing
//----------------------------------------------------------------------------
u08  ATAPI_DA_PauseResume(bool bResume)
{
	ATAPI_ClearBuf();
	minibuffer[0] = 0x4B;			// Command code (PAUSE)
	minibuffer[8] = bResume;		// 0 -> pause, 1 -> resume
	return ATAPI_Command();			// 0 if OK, else ERROR
}

//----------------------------------------------------------------------------
// Search function, skipping (CDDA_SCAN_STEP) seconds. 
// bDir=0 -> forward, bDir=1 -> backward
//----------------------------------------------------------------------------
void  ATAPI_DA_Scan(bool bDir)
{
	u08 err;
	if (ATAPI_DA_GetSubC() == 0)
	{
		u08 m = BUFFER1P[9];
		u08 s = BUFFER1P[10];
		if (bDir)
		{
			s -= CDDA_SCAN_STEP;
			if(((s08)s < 0) && m > 0)
			{
				s += 60;
				m--;
			}
			if ((s16)convert_to_ms(m,s) < 2)
				return;

		}
		else
		{
			s += CDDA_SCAN_STEP;
			if(s > 59)
			{
				s -= 60;
				m++;
			}
			if ((s16)convert_to_ms(m,s) > (s16)ATAPI_DA_GetDiscTime())
				return;

		}
		do{
			err = ATAPI_DA_PlayMSF(m, s, 0);
			delayms(100 * DELAY_TIME);
			err |= ATAPI_DA_GetSubC();
		} while (err);
	}
}

//----------------------------------------------------------------------------
// Get SubChannel Data to BUFFER1 memory (buffers is not used in CDDA mode)
//----------------------------------------------------------------------------
u08  ATAPI_DA_GetSubC(void)
{
	u08 err;
	CheckBusy();
	BUFFER1P[1] = 0xFF;
	ATAPI_ClearBuf();
	minibuffer[0] = 0x42;			// Command code (Read SUBCHANNEL)
	minibuffer[1] = 0x02;
	minibuffer[2] = 0x40;
	minibuffer[3] = 0x01;			// 01 - current position
	minibuffer[7] = 0x08;			// 2 kB
	err = GetDataFromCD(BUFFER1P);		// Get data from CD
	delayms(2);

	if(BUFFER1P[1] == 0x11)
		return 0;
	else
		return (BUFFER1P[1]);
}


//----------------------------------------------------------------------------
// Get Current Time
//----------------------------------------------------------------------------
u16  ATAPI_DA_GetCurTime(void)
{
	return( convert_to_ms(BUFFER1P[13] , BUFFER1P[14]) );
}

//----------------------------------------------------------------------------
// Get Current Track Number
//----------------------------------------------------------------------------
u08  ATAPI_DA_GetCurTrack(void)
{
	return (BUFFER1P[6] - 1);
}



// ***************************************************************************
// 			DATA CD AND GENERAL FUNCTIONS
// ***************************************************************************

u32 ATAPI_GetIsoToc(void)		// get last session TOC in multisession disc
{
	u08 err;
	union u32convert LBAC;

	LBAC.value = 0;
	DIR_CACHE[5] = 0xFF;				// clear disc type field
	ATAPI_ClearBuf();					// clear command buffer
	minibuffer[0]=0x43;					// READ TOC command, LBA Format
	minibuffer[2]=0x01;					// Multisession mode
	minibuffer[7]=0x08;					// 2 kB
	err = GetDataFromCD(DIR_CACHE);		// Get data from CD

	u16 LsDescAddr = 8* (DIR_CACHE[3] - 1) + 4;		// address of last session data descriptor
	if ((DIR_CACHE[LsDescAddr + 1] & 0x0d) == 4)		// if this session is digital data disc 
	{
		LBAC.bytes.byte4 = DIR_CACHE[LsDescAddr + 4];	// get bootblock adddrss
		LBAC.bytes.byte3 = DIR_CACHE[LsDescAddr + 5];
		LBAC.bytes.byte2 = DIR_CACHE[LsDescAddr + 6];
		LBAC.bytes.byte1 = DIR_CACHE[LsDescAddr + 7];
	}
	return LBAC.value;
}

//----------------------------------------------------------------------------
// Read one 2048 bytes sector in LBA mode from CD
// Returns 0 if no error detected
//----------------------------------------------------------------------------
u08 ATAPI_ReadLBA(u32 LBA, u08 *Buffer)
{
	union u32convert LBAC;
	LBAC.value = LBA;				// Prepare parameters...
	ATAPI_ClearBuf();
	minibuffer[0]=ATAPI_READ12;			// Read(12) Command
	minibuffer[2]=LBAC.bytes.byte4;
	minibuffer[3]=LBAC.bytes.byte3;
	minibuffer[4]=LBAC.bytes.byte2;
	minibuffer[5]=LBAC.bytes.byte1;
	minibuffer[9]=1;	
	return (GetDataFromCD(Buffer));
}

extern u08 cdda;

u08 GetDataFromCD(u08 *Buffer)
{
	u08 i,s;
	u08 j=0;

CMDIN:	i = 0;
	while(ATAPI_Command() && (++j < 5)); //  PRINT("\r\nCmdErr");	// Issue ATAPI Packet command
	delayms((cdda) ? 15 : 3);

	if (++j > 5)
	{
		ATA_SW_Reset();
//		PRINT("\r\nATAPI Reset");
		delayms(1);
		j = CD_STAT_UNDEF_ERROR;
		return (j);					// Too more errors > EXIT
	}

	if ((s=CheckDataReq()))					// If error
	{
		i = ReadBYTE(CMD_A | CMD_ERROR);
//		PRINT("\r\nATAPI Error: "); UART_Printfu08(s); UART_SendByte(32);
//		UART_Printfu08(i); UART_SendByte(32); UART_Printfu08(j);
		if (i == 0x20)
		{
			j = CD_STAT_NODISK;				// Media not present > EXIT
			return (j);
		}
/*
		if (i==0xD0 && j>0)
		{
			j--;
			goto CMDIN;
		}
*/
		if (s == 2)
			delayms(10);

		goto CMDIN;
	}
	cli();		

	i = (minibuffer[0] == ATAPI_REQUEST_SENSE || minibuffer[0] == ATAPI_MODE_SENSE) ? 1 : BPS/MINIBUFFERSIZE;

	do								// cycle sector by minibuffers 
	{
		if(i-- == 0)						// Check count of sended 128 bytes blocks
		{
//			PRINT("\r\nTo More data !!");
			sei();
			return(1);					// CD-ROM try send to more data
		}

		SetAddress(CMD_A + CMD_DATA);			// setup addressing and chip selects
		DDRA = 0;					// port A as input
		DDRC = 0;					// port C as input
		PORTA = 0xff;					// activate Pullups
		PORTC = 0xff;					// activate Pullups

		u08 *miniptr = minibuffer;
		j = MINIBUFFERSIZE;
		do{
			PORTB &= ~ 2;				// set DIOR lo
			j-=2;					// nop replacement, allow pin change
			*miniptr++ = PINA;			// read lo byte	
			*miniptr++ = PINC;			// read hi byte
			PORTB |= 2;				// set DIOR hi
		}while(j > 0);

		enable_ram();					// enable ExtRAM

		miniptr = minibuffer;
		j = MINIBUFFERSIZE;
		do{
			*Buffer++ = *miniptr++;
		}while(--j > 0);

	} while ((ReadBYTE(CMD_A | CMD_STATUS) & (SR_BUSY | SR_DRQ)) == SR_DRQ);

	ReadBYTE(CTRL_A | CTRL_ALT_STAT);
	sei();							// enable interrupt
	CheckBusy();
	return(ReadBYTE(CMD_A | CMD_STATUS) & SR_ERR);		// return 0 if OK, 1 if error
}

//----------------------------------------------------------------------------
// Select address and CS signals
//----------------------------------------------------------------------------
void SetAddress(u08 addr)
{
	MCUCR &= ~ _BV(SRE);		// Disable ExtRAM
	DDRA = 0xff;
	PORTA = addr;
	PORTE |= _BV(PE1);
	PORTE &= ~ _BV(PE1);
}


//----------------------------------------------------------------------------
// Read data BYTE from Drive
//----------------------------------------------------------------------------
u08 ReadBYTE(u08 addr)
{
register u08 tmp;
	SetAddress(addr);
	DDRA = 0;			// port A as input
	DDRC = 0;			// port C as input
	PORTA = 0xff;			// activate Pullups
	asm volatile ("nop");		// allow pin change
	PORTB &= ~ 2;			// set DIOR lo
	asm volatile ("nop");		// allow pin change
	tmp = PINA;			// read byte
	PORTB |= 2;			// set DIOR hi
	enable_ram();			// enable RAM
  	return tmp;
}


//----------------------------------------------------------------------------
// Write data BYTE to Drive
//----------------------------------------------------------------------------
void WriteBYTE(u08 addr, u08 dat)
{
  	SetAddress(addr);
	DDRC = 0xff;			// port C as output
	PORTA = dat;			// write byte
	PORTC = 0;
	asm volatile ("nop");		// allow pin change
	PORTB &= ~ 1;			// set DIOW lo
	asm volatile ("nop");		// allow pin change
	PORTB |= 1;			// set DIOW hi
	enable_ram();			// enable RAM
}

//----------------------------------------------------------------------------
// Send the PACKET command to CD-ROM and 12 command data bytes from minibuffer
//----------------------------------------------------------------------------
u08 ATAPI_Command()
{
	register u08 i = 0;
	CheckBusy();
	WriteBYTE(CMD_A | CMD_FEATURES , 0);
	WriteBYTE(CMD_A | CMD_CYL_HIGH , 8);			// Max transfer length
	WriteBYTE(CMD_A | CMD_CYL_LOW ,  0);			// = 2048 bytes
	WriteBYTE( CMD_A | CMD_COMMAND, ATAPI_PACKET );		// Write command Packet
	delay10us();
	CheckBusy();
	cli();
  	SetAddress(CMD_A | CMD_DATA);
	DDRC = 0xff;						// port C as output
	u08 *miniptr = minibuffer;
	i=6;							// 6 words to transfer = 12 bytes cmd
	do{
		PORTA = *miniptr++;
		PORTC = *miniptr++;
		asm volatile ("nop");				// allow pin change
		PORTB &= ~ 1;					// set DIOW lo
		asm volatile ("nop");				// allow pin change
		PORTB |= 1;					// set DIOW hi
	}while(--i > 0);

	enable_ram();						// enable RAM
	delay10us();
	ReadBYTE(CTRL_A | CTRL_ALT_STAT);
	sei();
	CheckBusy();
	return (ReadBYTE(CMD_A | CMD_STATUS) & SR_ERR);		// Return command status
}


//----------------------------------------------------------------------------
// Check CD-ROM status
//----------------------------------------------------------------------------
u08 ATAPI_CheckStatus(void)
{
	ATAPI_ClearBuf();
	minibuffer[0] = ATAPI_MODE_SENSE;			// Request Mode Sense
	minibuffer[2] = 0x2A;					// Mechanical status page
	minibuffer[8] = 0x18;
	GetDataFromCD(SENSE_BUF);
	if(SENSE_BUF[2] == 0x71)
		return CD_STAT_OPENED;

	ATAPI_ClearBuf();
	ATAPI_Command();					// Execute TEST UNIT READY command
	delayms(1);

	ATAPI_ClearBuf();
	minibuffer[0] = ATAPI_REQUEST_SENSE;			// Request Sense for TEST UNIT READY command
	minibuffer[4] = 18;					// Allocation length
	GetDataFromCD(SENSE_BUF);

/*
		u08 i;
		for (i=0; i<20; i++)
		{
			UART_Printfu08(SENSE_BUF[i]);
			UART_SendByte(32);
		}
		EOL();
*/
	switch (SENSE_BUF[12])
	{
		default:	return CD_STAT_UNDEF_ERROR;
		case 0x29:
		case 0x04:	return CD_STAT_NOT_READY;
		case 0x3A:	return CD_STAT_NODISK;
		case 0x00:	return CD_STAT_OK;
	}
	return CD_STAT_OK;
}

//----------------------------------------------------------------------------
// Slow down CD-ROM unit
//----------------------------------------------------------------------------
void SlowDown()
{
#ifdef READ_SPEED
	ATAPI_ClearBuf();
	minibuffer[0] = ATAPI_SLOWDOWN;
	minibuffer[2] = (READ_SPEED)>>8;
	minibuffer[3] = (u08)(READ_SPEED);
	ATAPI_Command();
#endif
}

//----------------------------------------------------------------------------
// Load or Unload CD
//----------------------------------------------------------------------------
void CD_Cmd(u08 Funct)
{
	ATAPI_ClearBuf();
	minibuffer[0] = ATAPI_START_STOP_UNIT;
	minibuffer[1] = 0x01;			// Immed = 1
	minibuffer[4] = Funct;
	ATAPI_Command();
}

//----------------------------------------------------------------------------
// Check ATA busy status
//----------------------------------------------------------------------------
u08 CheckBusy(void)
{
	u16 tout = 0;
	wdt_reset();
	while (((ReadBYTE(CMD_A | CMD_STATUS) & SR_BUSY) == SR_BUSY) && (++tout < 10000))
		delay10us();
	return (tout < 10000) ? 0:1;
}

//----------------------------------------------------------------------------
// Check ATA data request status, 0=OK, 1=ERROR, 2=Timeout
//----------------------------------------------------------------------------
u08 CheckDataReq(void)
{
	u16 timeout = 0;
	u16 maxtout = 500 + (50000 * (bPlaying == 0));
	register u08 stat;
	wdt_reset();
	while(1)
	{
		stat = ReadBYTE(CMD_A | CMD_STATUS);
		if (stat & SR_ERR) 
		{
//			PRINT("\r\nDREQ ERROR");
			return 1;
		}
		if (++timeout > maxtout) 
		{
//			PRINT("\r\nDREQ Timeout");
			return 2;
		}
		if ((stat & (SR_BUSY | SR_DRQ)) == SR_DRQ)
			return 0;
		delay10us();
	}
}
