#include "MKL03Z4.h"
#include "main.h"
#include "hardware.h"
#include "max6950.h"
#include "rx8900.h"
#include "lis2dw12.h"

uint8_t events, state; 
uint16_t displayCnt;
uint8_t shiftReg1, shiftReg2, butState, butFlag;// variables for buttons debouncing
uint8_t beepCnt, beepStat;

int main(void)
{
	SMC_setup();								// configure low-power modes
	RTC_setup();
	PORTS_setup();								// I/O ports setup	
	I2C0_setup();								// set up I2C0 module
	SPI0_setup();								// set up SPI0 module
	TPM0_setup();
	LPTMR0_setup();								// set up low-power timer
	delay(500);									// power-up delay
	RX8900_setup();
	LIS2DW12_setup();
	events = state = displayCnt = 0;
	
	PORTB->PCR[IRQ] |= PORT_PCR_ISF_MASK | PORT_PCR_IRQC(10);	// enable ACC interrupt	
	PORTB->PCR[ALARM] |= PORT_PCR_ISF_MASK | PORT_PCR_IRQC(10);	// enable RTC interrupt
	
	while(1)
	{		
		SCB->SCR = SCB_SCR_SLEEPDEEP_Msk;		// enable deep sleep mode
		__WFI();								// wait for event	

		switch (state)
		{
			case 0:								// display off
			if (events)
			{		
				RX8900_getTime();				// read time from RTC
				Display_Time();					// load time to MAX6950				
				NVIC_ClearPendingIRQ(LPTMR0_IRQn);	// clear pending LPTRM0 interrupt
				LPTMR0_CMR = DEBOUNCE;				// set debounce delay in ms 
				LPTMR0_CSR = LPTMR_CSR_TCF_MASK | LPTMR_CSR_TIE_MASK | LPTMR_CSR_TEN_MASK; // start timer			
				displayCnt = 0;					// init display ON time counter 
				shiftReg1 = shiftReg2 = 0xFF;
				butState = 0xFF;
				if (events & RTC_event)
					alarmOn();
				else
					state = 1;
			}
			break;
			
			//------------------------------------------------------------------------------
			case 1:								// display time
			displayCnt++;	
			if (displayCnt == TIMEON)
			{
				LPTMR0_CSR = 0;					// stop display timer
				Display_Off();					// turn off display
				LIS2DW12_getInt();				// erase all ACC interrupt flags
				PORTB->PCR[IRQ] |= PORT_PCR_ISF_MASK | PORT_PCR_IRQC(10);	// enable ACC interrupt
				events = 0;						// clear all event
				state = 0;
			}
			else if (events & RTC_event)
					alarmOn();
			else
			{	
				debounce();
				if (butFlag & (1<<SELECT))		// SELECT button is pressed
				{
					LIS2DW12_getData();			// get ACC values
					if (ACC_data[6] <= 0x19 || ACC_data[6] >= 0xE7) // vertical position
					{	
						Send_MAX6950(0x6300 + seg_table[time[2] >> 4]);
						Send_MAX6950(0x6200 + seg_table[time[2] & 0x0F]);
						Send_MAX6950(0x6000 + acl[1]); 	// display CL
						Send_MAX6950(0x6100 + acl[2]);	// instead of minutes		
						state = 2;
					}
					else						// horizontal position - set alarm
					{
						Send_MAX6950(0x6300 + seg_table[alTime[2] >> 4]);
						Send_MAX6950(0x6200 + seg_table[alTime[2] & 0x0F]);
						Send_MAX6950(0x6000 + acl[0]); 	// display AL
						Send_MAX6950(0x6100 + acl[2]);	// instead of minutes		
						state = 4;		
					}
				}
			}
			break;
			
			//------------------------------------------------------------------------------
			case 2:								// set hours
			debounce();	
			if (butFlag & (1<<SET))				// SET button is pressed
			{
				time[2]++;						// increment hours
				if ((time[2] & 0x0F) == 0x0A)	// modulo 24
					time[2] += 0x06;
				if (time[2] >= 0x24)
					time[2] = 0x00;
				Send_MAX6950(0x6300 + seg_table[time[2] >> 4]); 	// display tens of hours
				Send_MAX6950(0x6200 + seg_table[time[2] & 0x0F]);	// display units of hours			
			}	
			else if (butFlag & (1<<SELECT))		// SELECT button is pressed
			{
				Send_MAX6950(0x6000 + seg_table[time[1] >> 4]);		// display tens of minutes
				Send_MAX6950(0x6100 + seg_table[time[1] & 0x0F]);	// display units of minutes
				Send_MAX6950(0x6300 + acl[1]); 	// display CL
				Send_MAX6950(0x6200 + acl[2]);	// instead of hours
				state = 3;
			}
			break;
			
			//------------------------------------------------------------------------------
			case 3:								// set minutes
			debounce();	
			if (butFlag & (1<<SET))				// SET button is pressed
			{
				time[1]++;						// increment minutes
				if ((time[1] & 0x0F) == 0x0A)	// modulo 60
					time[1] += 0x06;
				if (time[1] >= 0x60)
					time[1] = 0x00;
				Send_MAX6950(0x6000 + seg_table[time[1] >> 4]); 	// display tens of minutes
				Send_MAX6950(0x6100 + seg_table[time[1] & 0x0F]);	// display units of minutes			
			}	
			else if (butFlag & (1<<SELECT))		// SELECT button is pressed
			{
				Update_Time();					// update time on display
				RX8900_setTime();				// upload time to RTC chip
				displayCnt = 0;					// clear time display timeout
				state = 1;
			}				
			break;
			
			//------------------------------------------------------------------------------
			case 4:								// set alarm hours
			debounce();	
			if (butFlag & (1<<SET))				// SET button is pressed
			{
				alTime[2]++;					// increment hours
				if ((alTime[2] & 0x0F) == 0x0A)	// modulo 24
					alTime[2] += 0x06;
				if (alTime[2] >= 0x25)
					alTime[2] = 0x00;
				Send_MAX6950(0x6300 + seg_table[alTime[2] >> 4]); 	// display tens of hours
				Send_MAX6950(0x6200 + seg_table[alTime[2] & 0x0F]);	// display units of hours			
			}	
			else if (butFlag & (1<<SELECT))		// SELECT button is pressed
			{
				Send_MAX6950(0x6000 + seg_table[alTime[1] >> 4]);	// display tens of minutes
				Send_MAX6950(0x6100 + seg_table[alTime[1] & 0x0F]);	// display units of minutes
				Send_MAX6950(0x6300 + acl[0]); 	// display AL
				Send_MAX6950(0x6200 + acl[2]);	// instead of hours
				state = 5;
			}				
			break;
			
			//------------------------------------------------------------------------------
			case 5:								// set alarm minutes
			debounce();	
			if (butFlag & (1<<SET))				// SET button is pressed
			{
				alTime[1]++;					// increment minutes
				if ((alTime[1] & 0x0F) == 0x0A)	// modulo 60
				alTime[1] += 0x06;
				if (alTime[1] >= 0x60)
					alTime[1] = 0x00;
				Send_MAX6950(0x6000 + seg_table[alTime[1] >> 4]); 	// display tens of minutes
				Send_MAX6950(0x6100 + seg_table[alTime[1] & 0x0F]);	// display units of minutes			
			}	
			else if (butFlag & (1<<SELECT))		// SELECT button is pressed
			{
				Update_Time();					// display time on display
				RX8900_setAlarm();				// upload alarm time to RTC chip
				displayCnt = 0;					// clear time display timeout
				state = 1;
			}							
			break;
			
			//------------------------------------------------------------------------------
			case 6:								// alarm on
			displayCnt++;	
			debounce();
			if (butFlag || (displayCnt == ALRMON))
			{
				LPTMR0_CSR = 0;					// stop display timer
				Display_Off();					// turn off display
				beep(0);						// turn off buzzer
				LIS2DW12_getInt();				// erase all ACC interrupt flags
				PORTB->PCR[IRQ] |= PORT_PCR_ISF_MASK | PORT_PCR_IRQC(10);	// enable ACC interrupt
				events = 0;						// clear ACC event			
				state = 0;
			}
			else
			{
				if (--beepCnt == 0)
				{
					beepCnt = ALPERI;			// restore beeping period		
					beepStat ^= 1;				// flip beep status
					beep(beepStat);		
				}					
			}				
			break;	
			
		}
	}
}

void alarmOn()									// start alarm sound
{
	alTime[2] = 0x24;							// disable alarm in RTC
	RX8900_setAlarm();
	beepCnt = ALPERI;
	beepStat = 1;
	beep(beepStat);								// start beeper					
	state = 6;	
}

void beep(uint8_t sound)						// buzzer on/off					 
{
	if (sound) 
		TPM0->SC = TPM_SC_CMOD(1);				// start beeper 
	else	
		TPM0->SC = TPM_SC_CMOD(0);				// stop beeper
}

//---------------------------------------------------------------------
void LPTMR0_IRQHandler(void)
{
	LPTMR0_CSR |= LPTMR_CSR_TCF_MASK;			// clear timer interrupt
	NVIC_ClearPendingIRQ(LPTMR0_IRQn);
	SCB->SCR = 0;								// disable VLPS/LSS sleep mode
}

//---------------------------------------------------------------------
void PORTB_IRQHandler(void)
{
	if (PORTB->PCR[IRQ] & PORT_PCR_ISF_MASK)	// ACC event
	{	
		events |= ACC_event;
		PORTB->PCR[IRQ] = PORT_PCR_ISF_MASK | PORT_PCR_MUX(1) | PORT_PCR_PE_MASK | PORT_PCR_PS_MASK;; 	// disable IRQ pin interrupt
	}
	if (PORTB->PCR[ALARM] & PORT_PCR_ISF_MASK)	// RTC event
	{
		events |= RTC_event;
		PORTB->PCR[ALARM] |= PORT_PCR_ISF_MASK;	// clear RTC flag
	}
	NVIC_ClearPendingIRQ(PORTB_IRQn);
	SCB->SCR = 0;								// disable VLPS/LSS sleep mode	
}

//---------------------------------------------------------------------
void debounce()								// buttons debouncing
{
	butFlag = 0;							// mark all buttons as not pressed
	shiftReg1 = (shiftReg1 >> 1) | (((PTB->PDIR & (1<<SELECT)) << (7-SELECT))); // update shift register
	if (butState & 1)						// check state of button1																							// old button state = OFF
	{
		if (shiftReg1 <= ON_THRESHOLD)		// button is pressed
		{
			butState &= 0xFE;				// mark button as pressed	
			butFlag |= (1<<SELECT);			// set flag for SELECT button
		}
	}
	else if (shiftReg1 >= OFF_THRESHOLD)	// button is released
		butState |= 1;						// mark button as released	
	
	
	shiftReg2 = (shiftReg2 >> 1) | (((PTB->PDIR & (1<<SET)) << (7-SET))); // update shift register
	if (butState & 2)						// old button state = OFF
	{
		if (shiftReg2 <= ON_THRESHOLD)		// button is pressed
		{
			butState &= 0xFD;				// mark button as pressed	
			butFlag |= (1<<SET);			// set flag for button2
		}
	}
	else if (shiftReg2 >= OFF_THRESHOLD)	// button is released
		butState |= 2;						// mark button as released
}
